//---------------------------------------------------------------------------
// Copyright Mark Pickersgill 2001-2014
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// File: WinScreenInterface.cpp
// Desc.: Implementation file for the Windows based ScreenInterface class.
//
// NOTE: The classes implemented in this file are specific to Win32
//---------------------------------------------------------------------------
// Created: 19/08/2001
// Version History:
//    19/08/2001, 0.0.0 - Created the file
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// Required CUSTOM header files
//---------------------------------------------------------------------------
#include "WinScreenInterface.h"
#include "TScreenSetupForm.h"
#include "WinScreenInterface.rh"
//---------------------------------------------------------------------------
// Required SYSTEM header files
//---------------------------------------------------------------------------
//#include <vcl.h>
//#pragma hdrstop
//#pragma package(smart_init)
#include <inifiles.hpp>
#include <memory>
//---------------------------------------------------------------------------
// Declarations
//---------------------------------------------------------------------------
IDData colourList [] =
{
{IDS_IBM_BLACK, IBM_BLACK},
{IDS_IBM_BLUE, IBM_BLUE},
{IDS_IBM_GREEN, IBM_GREEN },
{IDS_IBM_CYAN, IBM_CYAN},
{IDS_IBM_RED, IBM_RED },
{IDS_IBM_MAGENTA, IBM_MAGENTA },
{IDS_IBM_BROWN, IBM_BROWN },
{IDS_IBM_LIGHTGRAY, IBM_LIGHTGRAY },
{IDS_IBM_DARKGRAY, IBM_DARKGRAY },
{IDS_IBM_LIGHTBLUE, IBM_LIGHTBLUE },
{IDS_IBM_LIGHTGREEN, IBM_LIGHTGREEN },
{IDS_IBM_LIGHTCYAN, IBM_LIGHTCYAN },
{IDS_IBM_LIGHTRED, IBM_LIGHTRED },
{IDS_IBM_LIGHTMAGENTA, IBM_LIGHTMAGENTA },
{IDS_IBM_YELLOW, IBM_YELLOW },
{IDS_IBM_WHITE, IBM_WHITE },
{-1,-1}
};

//LBData c_controlList [] =
//{
//{IDS_NORMAL_BACK, CC_NORMAL_BACK, 0},
//{IDS_REVERSE, CC_REVERSE, 0},
//{IDS_NORMAL_FORE, CC_NORMAL_FORE, 0},
//{IDS_UNDERLINE, CC_UNDERLINE, 0},
//{IDS_ALT_FORE1, CC_ALT_FORE1, 0},
//{IDS_REV_UNDERLINE, CC_REV_UNDERLINE, 0},
//{IDS_ALT_FORE2, CC_ALT_FORE2, 0},
//{IDS_BRIGHT, CC_BRIGHT, 0},
//{IDS_ALT_FORE3, CC_ALT_FORE3, 0},
//{IDS_BRIGHT_BLINK, CC_BRIGHT_BLINK, 0},
//{IDS_ALT_BACK1, CC_ALT_BACK1, 0},
//{IDS_BLANK, CC_BLANK, 0},
//{IDS_ALT_BACK2, CC_ALT_BACK2, 0},
//{IDS_ALT_BACK3, CC_ALT_BACK3, 0},
//{IDS_BOX_LINE, CC_BOX_LINE, 0},
//{IDS_BLINK, CC_BLINK, 0},
//{-1,-1, 0}
//};

//---------------------------------------------------------------------------
// --== IMPLEMENTATION ==--
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// Class: WinScreenInterface
// Desc.: Windows Specific Screen Interface Class
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// Method: initVariables()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void WinScreenInterface::initVariables(TComponent* Owner)
{
_owner = Owner;
_screen = new TScreenControl(_owner);
_screen->Parent = dynamic_cast <TWinControl * > (_owner);
// Set the initial number of columns
setWidth(getWidth());
setHeight(getHeight());
calcFontNames(LANG_ENG, 640, 480, getWidth(), getHeight());
_screen->noScrollRegion(0, 0, 0, 1);
//_screen->TextBitmap->Canvas->Font->Size
//_screen->TextBitmap->Canvas->Font->Name = ScreenInterface::GetNormalFontName();
// Set focus of the new control
TCustomForm *ownerForm = dynamic_cast <TCustomForm * > (_owner);
if (ownerForm != 0)
   {
   ownerForm->ActiveControl = _screen;
   }
// Make the owner size the same size as the screen component
TControl *ownerControl = dynamic_cast <TControl * > (_owner);
ownerControl->ClientHeight = _screen->ClientHeight;
ownerControl->ClientWidth = _screen->ClientWidth;
}

//---------------------------------------------------------------------------
//  PUBLIC Methods
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// Method: WinScreenInterface()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
WinScreenInterface::WinScreenInterface(TComponent* Owner)
   : ScreenInterface()
{
initVariables(Owner);
}

//---------------------------------------------------------------------------
// Method: WinScreenInterface()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
WinScreenInterface::WinScreenInterface(TComponent* Owner, const int width, const int height)
   : ScreenInterface(width, height)
{
initVariables(Owner);
}

//---------------------------------------------------------------------------
// Method: WinScreenInterface()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
WinScreenInterface::WinScreenInterface(TComponent* Owner, const int width, const int height,
      const colour_t fore, const colour_t back)
   : ScreenInterface(width, height, fore, back)
{
initVariables(Owner);
}

//---------------------------------------------------------------------------
// Method: ~WinScreenInterface()
// Desc.: Destructor
// Return:
// Notes:
//---------------------------------------------------------------------------
WinScreenInterface::~WinScreenInterface()
{
delete _screen;
}

//---------------------------------------------------------------------------
// Method: fromWinColour()
// Desc.: Converts a windows colour into an IBM colour.
//  Only the basic colours are convertable, e.g. black, gray etc.
// Return:
// Notes:
//---------------------------------------------------------------------------
colour_t WinScreenInterface::fromWinColour(Graphics::TColor win_colour)
{
colour_t tmpcolour;

switch (win_colour)
    {
    case clBlack :
        tmpcolour = IBM_BLACK;
        break;
    case clNavy :
        tmpcolour = IBM_BLUE;
        break;
    case clGreen :
        tmpcolour = IBM_GREEN;
        break;
    case clTeal :
        tmpcolour = IBM_CYAN;
        break;
    case clMaroon :
        tmpcolour = IBM_RED;
        break;
    case clPurple :
        tmpcolour = IBM_MAGENTA;
        break;
    case clOlive :
        tmpcolour = IBM_BROWN;
        break;
    case clLtGray :
        tmpcolour = IBM_LIGHTGRAY;
        break;
    case clDkGray :
        tmpcolour = IBM_DARKGRAY;
        break;
    case clBlue :
        tmpcolour = IBM_LIGHTBLUE;
        break;
    case clLime :
        tmpcolour = IBM_LIGHTGREEN;
        break;
    case clAqua :
        tmpcolour = IBM_LIGHTCYAN;
        break;
    case clRed :
        tmpcolour = IBM_LIGHTRED;
        break;
    case clFuchsia :
        tmpcolour = IBM_LIGHTMAGENTA;
        break;
    case clYellow :
        tmpcolour = IBM_YELLOW;
        break;
    case clWhite :
        tmpcolour = IBM_WHITE;
        break;
    }
return (tmpcolour);
}

//---------------------------------------------------------------------------
// Method: toWinColour()
// Desc.: Converts an IBM colour code into a windows colour code.
// Return:
// Notes:
//---------------------------------------------------------------------------
Graphics::TColor WinScreenInterface::toWinColour(colour_t screen_colour)
{
TColor tmpcolour;

switch (screen_colour)
    {
    case IBM_BLACK :
        tmpcolour = clBlack; //RGB(  0,  0,  0);
        break;
    case IBM_BLUE :
        tmpcolour = clNavy; //RGB(  0,  0,128);
        break;
    case IBM_GREEN :
        tmpcolour = clGreen; //RGB(  0,128,  0);
        break;
    case IBM_CYAN :
        tmpcolour = clTeal; //RGB(  0,128,128);
        break;
    case IBM_RED :
        tmpcolour = clMaroon; //RGB(128,  0,  0);
        break;
    case IBM_MAGENTA :
        tmpcolour = clPurple; //RGB(128,  0,128);
        break;
    case IBM_BROWN :
        tmpcolour = clOlive; //RGB(128,128,  0);
        break;
    case IBM_LIGHTGRAY :
        tmpcolour = clLtGray;
        break;
    case IBM_DARKGRAY :
        tmpcolour = clDkGray; //RGB(128,128,128);
        break;
    case IBM_LIGHTBLUE :
        tmpcolour = clBlue; //RGB(  0,  0,255);
        break;
    case IBM_LIGHTGREEN :
        tmpcolour = clLime; //RGB(  0,255,  0);
        break;
    case IBM_LIGHTCYAN :
        tmpcolour = clAqua; //RGB(  0,255,255);
        break;
    case IBM_LIGHTRED :
        tmpcolour = clRed; //RGB(255,  0,  0);
        break;
    case IBM_LIGHTMAGENTA :
        tmpcolour = clFuchsia; //RGB(255,  0,255);
        break;
    case IBM_YELLOW :
        tmpcolour = clYellow; //RGB(255,255,  0);
        break;
    case IBM_WHITE :
        tmpcolour = clWhite; //RGB(255,255,255);
        break;
    }
return (tmpcolour);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void WinScreenInterface::setCaretOn(const bool state)
{
if (state != getCaretOn())
   {
   ScreenInterface::setCaretOn(state);
   _screen->winCaretOn(state);
   }
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
bool WinScreenInterface::getCaretOn(void) const
{
return (ScreenInterface::getCaretOn());
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
bool WinScreenInterface::clearScreen(void)
{
ScreenInterface::clearScreen();
_screen->clearGraphics();
redrawScreen(true);
return (true);
}

void WinScreenInterface::drawTextLine(const direction_t direction, bool erase)
{
int x1=0;
int y1=0;
int x2=0;
int y2=0;
int x=getx();
int y=gety();
unsigned char f;

if (erase)
   {
   f = 0x00;
   }
else
   {
   f = 0x01;
   }

charAttribs_t attribs;

switch (direction)
   {
   case DIR_UP :
   case DIR_DOWN :
      x1 = x * _screen->FontWidth + (_screen->FontWidth / 2);
      y1 = y * _screen->FontHeight;
      x2 = x1;
      y2 = y1 + _screen->FontHeight+1;  // Compensate for the LineTo() method
      setFlagsAt(x, y, f);
   break;
   case DIR_LEFT :
      // Draw a top line from the center of the previous character position to
      // the center of the current character position at the line below the
      // current line.
      x1 = x * _screen->FontWidth + (_screen->FontWidth / 2);
      y1 = (y+1) * _screen->FontHeight;
      x2 = x1 + _screen->FontWidth+1;  // Compensate for the LineTo() method
      y2 = y1;
      setFlagsAt(x, y+1, f);
      setFlagsAt(x+1, y+1, f);
   break;
   case DIR_RIGHT :
      // Draw a top line from the center of the previous character position
      // to the center of the current character position at the current line.
      x1 = (x-1) * _screen->FontWidth + (_screen->FontWidth / 2);
      y1 = y * _screen->FontHeight;
      x2 = x1 + _screen->FontWidth+1;  // Compensate for the LineTo() method
      y2 = y1;
      setFlagsAt(x, y, f);
      setFlagsAt(x-1, y, f);
   break;
   }
_screen->drawLine(x1, y1, x2, y2, erase, toWinColour(getColour(CC_BOX_LINE)));
// Store the extents of the box for later drawing to the canvas
if (_boxFlag == false)
   {
   _boxFlag = true;
   _boxX1 = x1;
   _boxY1 = y1;
   _boxX2 = x2;
   _boxY2 = y2;
   }
else
   {
   _boxY1 = min(_boxY1, y1); // This one is needed in case of downward scrolling
   _boxX2 = max(_boxX2, x2);
   _boxY2 = max(_boxY2, y2);
   }
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
bool WinScreenInterface::moveCaret(const direction_t direction, const int count)
{
//Check to see if we need to scroll
if ((direction == DIR_DOWN) && (gety() + count >= getHeight()-1))
   {
   scrollDown();
   return (true);
   }
else
   {
   ScreenInterface::moveCaret(direction, count);
   _screen->winSetCaretPos(getx(), gety());
   return (true);
   }
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void WinScreenInterface::setColour(c_control_t c_control, colour_t colour)
{
ScreenInterface::setColour(c_control, colour);
//The following should be implemented according to the FT2201 terminal spec
//, in the calling function
//if (c_control == CC_NORMAL_BACK)
//   {
//   ClearScreen();
//   }
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void WinScreenInterface::setDefColour(c_control_t c_control, colour_t colour)
{
ScreenInterface::setDefColour(c_control, colour);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void WinScreenInterface::defColoursToCur()
{
ScreenInterface::defColoursToCur();
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void WinScreenInterface::setStoredAttribs(const attrib_t &attribute)
{
ScreenInterface::setStoredAttribs(attribute);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void WinScreenInterface::getStoredAttribs(attrib_t &attribute) const
{
ScreenInterface::getStoredAttribs(attribute);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void WinScreenInterface::toggleAttribs(bool on)
{
ScreenInterface::toggleAttribs(on);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void WinScreenInterface::putChar(char character)
{
int xpos=getx(),
   ypos=gety();

ScreenInterface::putChar(character);
redrawRect(Rect(xpos, ypos, xpos, ypos), true);
moveCaret(DIR_RIGHT);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void WinScreenInterface::scrollDown(int lines)
{
int last_col = getWidth()-1,
   last_row = getHeight()-2;

ScreenInterface::scrollDown(lines);
// Scroll the bitmap
_screen->scroll(SD_DOWN, 1);
// Redraw the bottom lines of the bitmap
redrawRect(Rect(0, last_row-lines+1, last_col, last_row), false);
_screen->paintAll();
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void WinScreenInterface::setCharSet(charSet_t char_set)
{
ScreenInterface::setCharSet(char_set);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
charSet_t WinScreenInterface::getCharSet(void)
{
return (ScreenInterface::getCharSet());
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void WinScreenInterface::setRealUnderline(bool on)
{
ScreenInterface::setRealUnderline(on);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void WinScreenInterface::setRealReverse(bool on)
{
ScreenInterface::setRealReverse(on);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void WinScreenInterface::setRealBright(bool on)
{
ScreenInterface::setRealBright(on);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void WinScreenInterface::setRealBlink(bool on)
{
ScreenInterface::setRealBlink(on);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void WinScreenInterface::setDrawBoxLines(bool on)
{
ScreenInterface::setDrawBoxLines(on);
_screen->IgnoreGraphics = !on;
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
bool WinScreenInterface::getRealUnderline(void)
{
return (ScreenInterface::getRealUnderline());
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
bool WinScreenInterface::getRealReverse(void)
{
return (ScreenInterface::getRealReverse());
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
bool WinScreenInterface::getRealBright(void)
{
return (ScreenInterface::getRealBright());
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
bool WinScreenInterface::getRealBlink(void)
{
return (ScreenInterface::getRealBlink());
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void WinScreenInterface::clearStatusLine(void)
{
int top = getHeight()-1,
   bottom = top,
   right = getWidth()-1;

ScreenInterface::clearStatusLine();
// Update the screen
redrawRect(Rect(0, top, right, bottom), true);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void WinScreenInterface::setStatusLineAttribs(attrib_t attribs)
{
ScreenInterface::setStatusLineAttribs(attribs);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
bool WinScreenInterface::writeStatusLine(char character, bool scroll)
{
return (ScreenInterface::writeStatusLine(character, scroll));
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void WinScreenInterface::finishStatusLine(void)
{
int top = getHeight()-1,
   bottom = top,
   right = getWidth()-1;

ScreenInterface::finishStatusLine();
// Update the screen
redrawRect(Rect(0, top, right, bottom), true);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void WinScreenInterface::redrawRect(const TRect &rect, bool toscreen)
{
static fontSet_t curFontSet = FS_NORMAL;

if (validPosition(rect.Left, rect.Top) &&
   validPosition(rect.Right, rect.Bottom))
   {
   unsigned char character;
   unsigned char hex90 = (unsigned char)0x90;
   unsigned char hex80 = (unsigned char)0x80;
   bool use_underline = getRealUnderline();
   bool use_reverse = getRealReverse();
   bool use_blink = getRealBlink();
   bool use_bright = getRealBright();
   charAttribs_t attrib;
   TRect temprect=Rect(rect.Left*_screen->FontWidth, rect.Top*_screen->FontHeight,
            rect.Right*_screen->FontWidth+_screen->FontWidth, rect.Bottom*_screen->FontHeight+_screen->FontHeight);
   TRect charRect;   // Canvas rectangle for the current character

   for (int ypos=rect.Top; ypos <= rect.Bottom; ypos++)
      for (int xpos=rect.Left; xpos <= rect.Right; xpos++)
         {
         charRect = Rect(xpos * _screen->FontWidth-1, ypos * _screen->FontHeight-1,
            xpos * _screen->FontWidth+_screen->FontWidth+1, ypos * _screen->FontHeight+_screen->FontHeight+1);
         character = getCharAt(xpos, ypos, attrib);
         _screen->TextBitmap->Canvas->Font->Style = TFontStyles();
         //Check for Underline text
         if (use_underline && (attrib.Attrib2 == ATTR_UNDERLINE ||
            attrib.Attrib2 == ATTR_REV_UNDERLINE))
            _screen->TextBitmap->Canvas->Font->Style =
               _screen->TextBitmap->Canvas->Font->Style << fsUnderline;
         // Setup the colours
         _screen->TextBitmap->Canvas->Brush->Color =
            toWinColour(attrib.BackColour);
         _screen->TextBitmap->Canvas->Font->Color =
            toWinColour(attrib.ForeColour);

         if (character < hex80)
            {
            // Change the Font to either the Normal or Normal 2 font for
            // outputting the characters 0x0F to 0x7F
            if ((attrib.CharSet == CS_ALT2) && (curFontSet != FS_NORMAL2))
               {
               _screen->TextBitmap->Canvas->Font->Name = ScreenInterface::getNormalFontName2().data();
               curFontSet = FS_NORMAL2;
               }
            else if ((attrib.CharSet != CS_ALT2) && (curFontSet != FS_NORMAL))
               {
               _screen->TextBitmap->Canvas->Font->Name = ScreenInterface::getNormalFontName().data();
               curFontSet = FS_NORMAL;
               }
            }
         // Check for characters that are above HEX(90) in the Normal character
         // set, and set the character to an underlined version of the lower
         // value of the character
         else if ((character >= hex90) && (attrib.CharSet == CS_NORMAL))
            {
            if (curFontSet != FS_NORMAL)
               {
               // Change to the normal character set
               _screen->TextBitmap->Canvas->Font->Name = ScreenInterface::getNormalFontName().data();
               curFontSet = FS_NORMAL;
               }
            // Underline the character
            character = character - hex80;
            if (use_underline)
               _screen->TextBitmap->Canvas->Font->Style =
                  _screen->TextBitmap->Canvas->Font->Style << fsUnderline;
            else
               _screen->TextBitmap->Canvas->Font->Style =
                  _screen->TextBitmap->Canvas->Font->Style >> fsUnderline;
            }
         else if (character >= hex80)
            {
            // Change to the alternate character set
            switch (attrib.CharSet)
               {
               case CS_ALT2 :
                  _screen->TextBitmap->Canvas->Font->Name = ScreenInterface::getAlt2FontName().data();
                  curFontSet = FS_ALT2;
               break;
               case CS_ALT1 :
                  _screen->TextBitmap->Canvas->Font->Name = ScreenInterface::getAlt1FontName().data();
                  curFontSet = FS_ALT1;
               break;
               default :   // Default to the alternate0 character set
                  _screen->TextBitmap->Canvas->Font->Name = ScreenInterface::getAlt0FontName().data();
                  curFontSet = FS_ALT0;
               break;
               }
            }
         // Output the character to the Text Bitmap
         _screen->TextBitmap->Canvas->Brush->Style = bsSolid;
         _screen->TextBitmap->Canvas->TextOut(xpos * _screen->FontWidth,
            ypos * _screen->FontHeight, (char)character);

         // Paint the character to the final canvas
         if ((attrib.Flags & 0x01) == 0x01)
            {
            // This character is affected by box graphics, so include the box graphics in the painting
            _screen->paintRect(charRect, false, false);
            }
         else
            {
            // No Box graphics so we can draw directly to the Final Canvas
            _screen->FinalBitmap->Canvas->Font->Name = _screen->TextBitmap->Canvas->Font->Name;
            _screen->FinalBitmap->Canvas->Font->Style = _screen->TextBitmap->Canvas->Font->Style;
            _screen->FinalBitmap->Canvas->Font->Color = _screen->TextBitmap->Canvas->Font->Color;
            _screen->FinalBitmap->Canvas->Brush->Color = _screen->TextBitmap->Canvas->Brush->Color;
            _screen->FinalBitmap->Canvas->Brush->Style = bsSolid;
            _screen->FinalBitmap->Canvas->TextOut(xpos * _screen->FontWidth,
               ypos * _screen->FontHeight, (char)character);
            }
         }
   if (toscreen)
      {
      //Copy the updated region to the canvas
      //_screen->paintRect(temprect, false, true);
      _screen->copyRect(temprect);
      }
   }
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void WinScreenInterface::calcFontNames(language_t language, int resX, int resY,
   int screenX, int screenY)
{
ScreenInterface::calcFontNames(language, resX, resY, screenX, screenY);
// Calculate the font size. These sizes will be dependant on the Win32 platform
if ((resX <= 640) || (resY <= 480))
   {
   if (screenY <= 25)
      _fontSize = 13;
   else //if (screenY > 25)
      _fontSize = 10;
   }
else //if ((resX >= 800) && (resX < 1024) ||
   //(resY >= 600) && (resY < 768))
   {
   // Default to the heighest font size
   if (screenY <= 25)
      _fontSize = 17;
   else //if (screenY > 25)
      _fontSize = 12;
   }
// Get the new Font height and Width and store it
_screen->adjustForFont(ScreenInterface::getNormalFontName().data(), _fontSize);
redrawScreen(true);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void WinScreenInterface::redrawScreen(const bool toscreen)
{
redrawRect(Rect(0, 0, getWidth()-1, getHeight()-1), toscreen);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void WinScreenInterface::beep(void)
{
MessageBeep(MB_OK); //MB_ICONEXCLAMATION
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void WinScreenInterface::setWidth(const int width)
{
ScreenInterface::setWidth(width);
_screen->ScreenColumns = getWidth();
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void WinScreenInterface::setHeight(const int height)
{
ScreenInterface::setHeight(height);
_screen->ScreenLines = getHeight();
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
int WinScreenInterface::setup(void)
{
if (_owner != NULL)
   {
   // Display setup screen for the Printer
   std::auto_ptr<TScreenSetupForm> setupForm (new TScreenSetupForm(_owner));

   setupForm->foreColourCombo->load(colourList);
   setupForm->backColourCombo->load(colourList);
   setupForm->blinkColourCombo->load(colourList);
   setupForm->brightBlinkColourCombo->load(colourList);
   setupForm->boxColourCombo->load(colourList);
   setupForm->foreColourCombo->select(getDefColour(CC_NORMAL_FORE));
   setupForm->backColourCombo->select(getDefColour(CC_NORMAL_BACK));
   setupForm->blinkColourCombo->select(getDefColour(CC_BLINK));
   setupForm->brightBlinkColourCombo->select(getDefColour(CC_BRIGHT_BLINK));
   setupForm->boxColourCombo->select(getDefColour(CC_BOX_LINE));
   setupForm->chkDrawBoxLines->Checked = getDrawBoxLines();
   setupForm->rgFontSize->ItemIndex = _forcedFontSize;
   int result = setupForm->ShowModal();
   if (result == mrOk)
      {
      // Save the changes
      setDefColour(CC_NORMAL_FORE, (colour_t)setupForm->foreColourCombo->getSelected()); //setupForm->foreColourCombo->Objects[setupForm->foreColourCombo->ItemIndex]);
      setDefColour(CC_NORMAL_BACK, (colour_t)setupForm->backColourCombo->getSelected());
      setDefColour(CC_BLINK, (colour_t)setupForm->blinkColourCombo->getSelected());
      setDefColour(CC_BRIGHT_BLINK, (colour_t)setupForm->brightBlinkColourCombo->getSelected());
      setDefColour(CC_BOX_LINE, (colour_t)setupForm->boxColourCombo->getSelected());
      setDrawBoxLines(setupForm->chkDrawBoxLines->Checked);
      defColoursToCur();
      if (setupForm->rgFontSize->ItemIndex == 0)
         {
         calcFontNames(getLanguage(), 640, 480, getWidth(), getHeight());
         _forcedFontSize = 0;
         }
      else
         {
         calcFontNames(getLanguage(), 800, 600, getWidth(), getHeight());
         _forcedFontSize = 1;
         }
      _screen->adjustForFont(getNormalFontName().c_str(), _fontSize);
      }
   return result;
   }
else
   {
   return mrCancel;
   }
}

bool WinScreenInterface::loadSetup(const string &fileName)
{
try
   {
   std::auto_ptr<TIniFile> setupFile (new TIniFile(fileName.data()));
   _forcedFontSize = setupFile->ReadInteger("Screen", "FontSize", 0);
   if (_forcedFontSize == 0)
      calcFontNames(getLanguage(), 640, 480, getWidth(), getHeight());
   else
      calcFontNames(getLanguage(), 800, 600, getWidth(), getHeight());
   setDefColour(CC_NORMAL_FORE, (colour_t)setupFile->ReadInteger("Screen", "ForeColour", (unsigned int)IBM_LIGHTGRAY));
   setDefColour(CC_NORMAL_BACK, (colour_t)setupFile->ReadInteger("Screen", "BackColour", (unsigned int)IBM_BLACK));
   setDefColour(CC_BLINK, (colour_t)setupFile->ReadInteger("Screen", "BlinkColour", (unsigned int)IBM_LIGHTGREEN));
   setDefColour(CC_BRIGHT_BLINK, (colour_t)setupFile->ReadInteger("Screen", "BrightBlinkColour", (unsigned int)IBM_LIGHTCYAN));
   setDefColour(CC_BOX_LINE, (colour_t)setupFile->ReadInteger("Screen", "BoxColour", (unsigned int)IBM_YELLOW));
   setDrawBoxLines(setupFile->ReadBool("Screen", "DrawBoxLines", false));

   defColoursToCur();
   return (true);
   }
catch (...)
   {
   return (false);
   }
}

bool WinScreenInterface::saveSetup(const string &fileName)
{
try
   {
   std::auto_ptr<TIniFile> setupFile (new TIniFile(fileName.data()));
   setupFile->WriteInteger("Screen", "FontSize", _forcedFontSize);
   setupFile->WriteInteger("Screen", "ForeColour", getDefColour(CC_NORMAL_FORE));
   setupFile->WriteInteger("Screen", "BackColour", getDefColour(CC_NORMAL_BACK));
   setupFile->WriteInteger("Screen", "BlinkColour", getDefColour(CC_BLINK));
   setupFile->WriteInteger("Screen", "BrightBlinkColour", getDefColour(CC_BRIGHT_BLINK));
   setupFile->WriteBool("Screen", "DrawBoxLines", getDrawBoxLines());
   return (true);
   }
catch (...)
   {
   return (false);
   }
}

void WinScreenInterface::redraw()
{
ScreenInterface::redraw();
redrawRect(Rect(0, 0, getWidth()-1, getHeight()-1), true);
_screen->winSetCaretPos(getx(), gety());
}

//void WinScreenInterface::paintRect(const int x1, const int y1, const int x2, const int y2)
//{
//_screen->paintRect(Rect(x1*_screen->FontWidth, y1*_screen->FontHeight, x2*_screen->FontWidth, y2*_screen->FontHeight));
//_screen->winSetCaretPos(getx(), gety());
//}

void WinScreenInterface::beginDrawBox()
{
_boxFlag = false;
}

void WinScreenInterface::endDrawBox()
{
TRect r;
_boxFlag = false;
// Draw each side of the box
_screen->paintRect(Rect(_boxX1-1, _boxY1-1, _boxX2+1, _boxY1+1), false, false); // Top Line
_screen->paintRect(Rect(_boxX2-1, _boxY1, _boxX2+1, _boxY2), false, false);  // Right Vertical line
_screen->paintRect(Rect(_boxX1-1, _boxY2-1, _boxX2+1, _boxY2+1), false, false); // Bottom line
_screen->paintRect(Rect(_boxX1-1, _boxY1, _boxX1+1, _boxY2), false, false); // Left Vertical line
_screen->copyRect(Rect(_boxX1-1, _boxY1-1, _boxX2+1, _boxY2+1));
_screen->winSetCaretPos(getx(), gety());
}

int WinScreenInterface::getPixelWidth()
{
return _screen->TextBitmap->Width;
}

int WinScreenInterface::getPixelHeight()
{
return _screen->TextBitmap->Height;
}

//---------------------------------------------------------------------------
// END OF FILE
//---------------------------------------------------------------------------
