//---------------------------------------------------------------------------
// Copyright Mark Pickersgill 2001-2014
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// File: WinPrinterInterface.cpp
// Desc.: Implementation file for the Win32 specific WinPrinterInterface
//    class.
//
//---------------------------------------------------------------------------
// Created: 06/01/2002
// Version History:
//    06/01/2002, 0.0.0 - Created the file
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// Required CUSTOM header files
//---------------------------------------------------------------------------
#include "WinPrinterInterface.h"
#include "TPrinterSetupForm.h"
#include "WinPrinterInterface.rh"
//---------------------------------------------------------------------------
// Required SYSTEM header files
//---------------------------------------------------------------------------
#include <inifiles.hpp>
#include <memory>
//---------------------------------------------------------------------------
// Declarations
//---------------------------------------------------------------------------
// A table of the type of printer files that can be output and their related
// strings for display in the combo box
//---------------------------------------------------------------------------
// --== IMPLEMENTATION ==--
//---------------------------------------------------------------------------
const unsigned int DEF_TIMEOUT = 15;
//---------------------------------------------------------------------------
// Class: WinPrinterInterface
// Desc.: Win32 Specific Printer Interface Class
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// Method: WinPrinterInterface(owner)
// Desc.: Default Constructor
// Return: None
// Notes:
//---------------------------------------------------------------------------
WinPrinterInterface::WinPrinterInterface(TComponent *Owner) : PrinterInterface()
{
   _timeout = DEF_TIMEOUT;
   _timeCount = DEF_TIMEOUT;
   _owner = Owner;
   _fileName = new string();
   _defFileName = new string();
   _currentFile = new fstream();
   _dataType = new string("RAW");
   _defOutDevice = OUT_DEV_PRINTER;
   _outDevice = _defOutDevice;
   _prevOutDevice = OUT_DEV_FILE;
   _newPageId = new string("\x0C");
   _printerName = new string();
   _ignorePrintCodes = true;
   _useTransTable = false;
   _textPrinter = new TTextPrint(Owner);
   _textPrinter->Title = "EMU";
}

//---------------------------------------------------------------------------
// Method: ~WinPrinterInterface()
// Desc.: Destructor
// Return: None
// Notes:
//---------------------------------------------------------------------------
WinPrinterInterface::~WinPrinterInterface()
{
   close();
   _owner = 0;
   delete _fileName;
   delete _defFileName;
   delete _currentFile;
   delete _dataType;
   delete _newPageId;
   delete _printerName;
   delete _textPrinter;
}

//---------------------------------------------------------------------------
// Method:
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void WinPrinterInterface::setFileName(const string &newFileName)
{
   *_fileName = newFileName;
}

//---------------------------------------------------------------------------
// Method:
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void WinPrinterInterface::addToFileName(const char letter)
{
   *_fileName += letter;
}

//---------------------------------------------------------------------------
// Method:
// Desc.:
// Return:
// Notes: ### Other devices may be possible later
//---------------------------------------------------------------------------
void WinPrinterInterface::print(const char byte)
{
   _timeCount = 0;
   if (_outDevice == OUT_DEV_FILE)
      {
      // Write to the file
      if (!_currentFile->is_open())
         _currentFile->open(_fileName->data(), ios_base::out|ios_base::ate
            |ios_base::binary);
      if (_currentFile->is_open())
         {
         _currentFile->put(byte);
         _currentFile->flush();
         }
      }
   else
      {
      // Write to the printer
      if (!_textPrinter->Active)
      {
         _textPrinter->DataType = _dataType->data();
         _textPrinter->Open();
      }
      _textPrinter->Write(byte);

      // Write to the printer
      //### if byte == _newPageId then send the byte out and close the printer
      // that way it is released for other programs to use
   //   if (byte == (unsigned char)0x0C)  // ###Change
   //      {
   //      // We may have only been sent a new page marker
   //      if (!_textPrinter->Active)
   //         _textPrinter->Open();
   //      _textPrinter->Write(byte);
   //      //_textPrinter->Close();
   //      }
   //   else
   //      {
   //      if (!_textPrinter->Active)
   //         _textPrinter->Open();
   //      _textPrinter->Write(byte);
   //      }
      }
}

//---------------------------------------------------------------------------
// Method:
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void WinPrinterInterface::switchToPrinter(void)
{
   close();
   _outDevice = OUT_DEV_PRINTER;
//###Open a new print job
//
//if (_textPrinter->Open())
//   {
//   // Flush and close the file
//   _currentFile->close();
//   // Set the output device to printer
//   _outDevice = _prevOutDevice;
//   }
}

//---------------------------------------------------------------------------
// Method:
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void WinPrinterInterface::switchToFile(void)
{
   close();
   _outDevice = OUT_DEV_FILE;

// Open the file
// Set the output device to file
//_currentFile->close();
//_currentFile->open(_fileName->data(), ios_base::out|ios_base::ate
//   |ios_base::binary);
//if (_currentFile->is_open())
//   {
//   // Close the printer
//   _textPrinter->Close();
//   _prevOutDevice = _outDevice; // ###May have to specify the Printer directly
//   _outDevice = OUT_DEV_FILE;
//   }
}

//---------------------------------------------------------------------------
// Method:
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void WinPrinterInterface::close()
{
   _timeCount = _timeout;
   _currentFile->close();
   _textPrinter->Close();
}

//---------------------------------------------------------------------------
// Method:
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
bool WinPrinterInterface::checkTimeout(long int increment)
{
   if ((_outDevice == OUT_DEV_FILE && _currentFile->is_open()) ||
       (_outDevice == OUT_DEV_PRINTER && _textPrinter->Active))
   {
      _timeCount += increment;
      if (_timeCount >= _timeout)
      {
         close();
         return false;
      }
      return true;
   }
   else
      return false;
}

//---------------------------------------------------------------------------
// Method:
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
int WinPrinterInterface::setup(void)
{
   if (_owner != NULL)
   {
      // Display setup screen for the Printer
      auto_ptr<TPrinterSetupForm> setupForm (new TPrinterSetupForm(_owner));
      // Set the component values to the values within the object
      setupForm->edtDataType->Text = _dataType->data();
      setupForm->ignorePrintCodes->Checked = _ignorePrintCodes;
      setupForm->useTransTable->Checked = _useTransTable;
      setupForm->printerList->Items->Assign(_textPrinter->Printers);
      setupForm->printerList->ItemIndex = setupForm->printerList->Items->IndexOf(_printerName->data());
      setupForm->edtTimeout->Value = _timeout;
      //setupForm->newPageId->Text = toHexString(_newPageId);
      if (_outDevice == OUT_DEV_PRINTER)
         setupForm->cPrintToFile->Checked = false;
      else
         setupForm->cPrintToFile->Checked = true;
      if (_defOutDevice == OUT_DEV_PRINTER)
         setupForm->printToFile->Checked = false;
      else
         setupForm->printToFile->Checked = true;
      setupForm->cFileName->Text = _fileName->data();
      setupForm->fileName->Text = _defFileName->data();
      int result = setupForm->ShowModal();
      if (result == mrOk)
      {
         // Save the changes made
         _timeout = setupForm->edtTimeout->Value;
         _ignorePrintCodes = setupForm->ignorePrintCodes->Checked;
         _useTransTable = setupForm->useTransTable->Checked;
         //_newPageId = fromHexString(setupForm->newPageId->Text);
         *_dataType = setupForm->edtDataType->Text.c_str();
         if (*_dataType == "")
            *_dataType = "RAW";
         // Change the printer name cleanly
         *_printerName = setupForm->printerList->Items->Strings[setupForm->printerList->ItemIndex].c_str();
         _textPrinter->Printer = _printerName->data();
         _textPrinter->Close();
         // Default settings
         if (setupForm->printToFile->Checked)
            _defOutDevice = OUT_DEV_FILE;
         else
            _defOutDevice = OUT_DEV_PRINTER;
         *_defFileName = setupForm->fileName->Text.c_str();
         // Current settings
         *_fileName = setupForm->cFileName->Text.c_str();
         if (setupForm->cPrintToFile->Checked)
            switchToFile();
         else
            switchToPrinter();
      }
      return (result);
   }
   else
      return (mrCancel);
}

//---------------------------------------------------------------------------
// Method:
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
bool WinPrinterInterface::loadSetup(const string &fileName)
{
   try
   {
      auto_ptr<TIniFile> setupFile (new TIniFile(fileName.data()));
      *_dataType = setupFile->ReadString(PRINTER_SECTION, "DataType", "RAW").c_str();
      *_printerName = setupFile->ReadString(PRINTER_SECTION, "Printer Name", "").c_str();
      if (*_printerName == "")
         {
         // Get the default printer name
         *_printerName = _textPrinter->DefaultPrinter.c_str();
         }
      // Ensure the textPrinter object is kept up to date as well
      _timeout = setupFile->ReadInteger(PRINTER_SECTION, "CloseTimeout", DEF_TIMEOUT);
      _textPrinter->Printer = _printerName->data();
      *_defFileName = setupFile->ReadString(PRINTER_SECTION, "Print File Name", "").c_str();
      *_fileName = *_defFileName;
      _defOutDevice = (outDev_t)setupFile->ReadInteger(PRINTER_SECTION, "Output Device", OUT_DEV_PRINTER);
      _outDevice = _defOutDevice;
      if (_outDevice == OUT_DEV_FILE)
         {
         switchToFile();
         }
      _prevOutDevice = (outDev_t)setupFile->ReadInteger(PRINTER_SECTION, "Previous Output Device", OUT_DEV_FILE);

   //   *_newPageId = fromHexString(setupFile->ReadString(PRINTER_SECTION, "New Page ID", "")).c_str();
      _ignorePrintCodes = setupFile->ReadBool(PRINTER_SECTION, "Ignore Print Codes", true);
      _useTransTable = setupFile->ReadBool(PRINTER_SECTION, "Use Translation Table", false);
      return (true);
   }
   catch (...)
   {
      return (false);
   }
}

//---------------------------------------------------------------------------
// Method:
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
bool WinPrinterInterface::saveSetup(const string &fileName)
{
   try
   {
      auto_ptr<TIniFile> setupFile (new TIniFile(fileName.data()));
      setupFile->WriteString(PRINTER_SECTION, "DataType", _dataType->data());
      setupFile->WriteString(PRINTER_SECTION, "Printer Name", _printerName->data());
      setupFile->WriteInteger(PRINTER_SECTION, "CloseTimeout", _timeout);
      setupFile->WriteString(PRINTER_SECTION, "Print File Name", _defFileName->data());
      setupFile->WriteInteger(PRINTER_SECTION, "Output Device", (outDev_t)_defOutDevice);
      setupFile->WriteInteger(PRINTER_SECTION, "Previous Output Device", (outDev_t)_prevOutDevice);
   //   setupFile->WriteString(PRINTER_SECTION, "New Page ID", toHexString(_newPageId->data()));
      setupFile->WriteBool(PRINTER_SECTION, "Ignore Print Codes", _ignorePrintCodes);
      setupFile->WriteBool(PRINTER_SECTION, "Use Translation Table", _useTransTable);
      return (true);
   }
   catch (...)
   {
      return (false);
   }
}
//---------------------------------------------------------------------------
// END OF FILE
//---------------------------------------------------------------------------

