//---------------------------------------------------------------------------
// Copyright Mark Pickersgill 2001-2014
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// File: WinFileInterface.cpp
// Desc.: Implementation file for the Windows based FileInterface class.
//
// NOTE: The classes implemented in this file are specific to Win32
//---------------------------------------------------------------------------
// Created: 17/08/2001
// Version History:
//    17/08/2001, 0.0.0 - Created the file
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// Required CUSTOM header files
//---------------------------------------------------------------------------
#include "WinFileInterface.h"
//---------------------------------------------------------------------------
// Required SYSTEM header files
//---------------------------------------------------------------------------
#include <vcl.h>
#pragma hdrstop
#pragma package(smart_init)

//---------------------------------------------------------------------------
// --== IMPLEMENTATION ==--
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// Class: WinFileInterface
// Desc.: Windows Specific File Interface Class
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// Method: WinFileInterface()
// Desc.: Default Constructor
// Return: None
// Notes: change later to include application handle
//---------------------------------------------------------------------------
WinFileInterface::WinFileInterface() : FileInterface()
{
_fileName = new string();
_currentFile = new fstream();
}

//---------------------------------------------------------------------------
// Method: ~WinFileInterface()
// Desc.: Default Destructor
// Return: None
// Notes: 
//---------------------------------------------------------------------------
WinFileInterface::~WinFileInterface()
{
// Ensure the file is flushed and closed nicely
_currentFile->close();
delete _fileName;
delete _currentFile;
}

//---------------------------------------------------------------------------
// Method: addToFileName(letter)
// Desc.: Adds a letter to the internal filename
// Return: None
// Notes: 
//---------------------------------------------------------------------------
void WinFileInterface::addToFileName(const char letter)
{
*_fileName += letter;
}

//---------------------------------------------------------------------------
// Method: setFileName(fileName)
// Desc.: Sets the internal filename to the new filename
// Return: None
// Notes: 
//---------------------------------------------------------------------------
void WinFileInterface::setFileName(const string &fileName)
{
*_fileName = fileName;
}

//---------------------------------------------------------------------------
// Method: create()
// Desc.: Creates a new file on the disk and overwrites the file if one 
//    exists.
// Return: '0' on failure and '1' on success
// Notes: 
//---------------------------------------------------------------------------
char WinFileInterface::create(void)
{
ofstream newFile;

newFile.open(_fileName->c_str(), ios_base::trunc|ios_base::out);
if (newFile.is_open())
   {
   // Successfully created a new file
   newFile.close();
   return '1';
   }
else
   {
   // Failed on creating a new file
   return '0';
   }
}

//---------------------------------------------------------------------------
// Method: exists()
// Desc.: Checks to see if a file exists on the local disk
// Return: '0' on failure and '1' on success
// Notes: 
//---------------------------------------------------------------------------
char WinFileInterface::exists(void)
{
ifstream inFile;

inFile.open(_fileName->c_str(), ios_base::in|ios_base::nocreate);
if (inFile.is_open())
   {
   inFile.close();
   return '1'; // File exists
   }
else
   {
   return '0'; // File doesn't exist
   }
}

//---------------------------------------------------------------------------
// Method: open()
// Desc.: Opens the file for appending, create file if it doesn't already 
//    exist.
// Return: '0' on failure and '1' on success
// Notes: 
//---------------------------------------------------------------------------
char WinFileInterface::open(void)
{
_currentFile->open(_fileName->data(), ios_base::out|ios_base::ate
   |ios_base::binary);
if (_currentFile->is_open())
   {
   return '1'; // File opened correctly
   }
else
   {
   return '0'; // File did not open correctly
   }
}

//---------------------------------------------------------------------------
// Method: write(buffer, count)
// Desc.: Writes data to the opened file. If the file is not open this method
//    should not fail.
// Return: None
// Notes:
//---------------------------------------------------------------------------
void WinFileInterface::write(const char *dataBuffer, int size)
{
if (_currentFile->is_open() && (dataBuffer != NULL))
   {
   _currentFile->write(dataBuffer, size);
   }
}

//---------------------------------------------------------------------------
// Method: put(character)
// Desc.: Writes a character to the opened file. If the file is not open
//    this method should not fail.
// Return: None
// Notes:
//---------------------------------------------------------------------------
void WinFileInterface::put(const char character)
{
if (_currentFile->is_open())
   {
   _currentFile->put(character);
   }
}

//---------------------------------------------------------------------------
// Method: flush()
// Desc.: Flushes the data to disk so that the file maybe read without having
//    to close down the program
// Return: None
// Notes:
//---------------------------------------------------------------------------
void WinFileInterface::flush(void)
{
_currentFile->flush();
}
//---------------------------------------------------------------------------
// Method: close()
// Desc.: Closes the file opened previously and clears the filename string
// Return: None
// Notes:
//---------------------------------------------------------------------------
void WinFileInterface::close(void)
{
_currentFile->close();
_fileName->clear();
}

//---------------------------------------------------------------------------
// END OF FILE
//---------------------------------------------------------------------------
