//---------------------------------------------------------------------------
// Copyright Mark Pickersgill 2001-2014
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// File: WinCommInterface.cpp
// Desc.: Implementation file for the Win32 specific communications port
//    object
//
// NOTE: The classes implemented in this file are specific to Win32
//---------------------------------------------------------------------------
// Created: 17/09/2001
// Version History:
//    17/09/2001, 0.0.0 - First skeleton Implementation
//
// SEE HEADER FILE FOR FULL VERSION HISTORY
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// Required CUSTOM header files
//---------------------------------------------------------------------------
#include "WinCommInterface.h"
#include "TComSetupForm.h"
//---------------------------------------------------------------------------
// Required SYSTEM header files
//---------------------------------------------------------------------------
#include <memory>
//---------------------------------------------------------------------------
// --== IMPLEMENTATION ==--
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// Supporting Functions
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// Class:
// Desc.:
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// PRIVATE Methods
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void __fastcall WinCommInterface::_onReceive(TObject *Sender, int Count)
{
   if (CommInterface::onComReceive != 0)
   {
      CommInterface::onComReceive(Count);
   }
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void __fastcall WinCommInterface::_onOpen(TObject *Sender)
{
   if (CommInterface::onComOpen != 0)
      CommInterface::onComOpen();
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void __fastcall WinCommInterface::_onClose(TObject *Sender)
{
   if (CommInterface::onComClose != 0)
      CommInterface::onComClose();
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void __fastcall WinCommInterface::_on80Full(TObject *Sender)
{
   if (CommInterface::onCom80Full != 0)
      CommInterface::onCom80Full();
}

//---------------------------------------------------------------------------
// PUBLIC Methods
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// Method: WinCommInterface()
// Desc.: Default constructor
// Return: none
// Notes:
//---------------------------------------------------------------------------
WinCommInterface::WinCommInterface(TComponent *Owner) : CommInterface()
{
   _owner = Owner;
   _comPort = new TComPort(_owner);
   _comPort->Port = "COM1";
   setDefaults();
   _comPort->OnRxChar = _onReceive;
   _comPort->OnAfterOpen = _onOpen;
   _comPort->OnAfterClose = _onClose;
   _comPort->OnRx80Full = _on80Full;
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
WinCommInterface::~WinCommInterface()
{
   delete _comPort;
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void WinCommInterface::setDefaultFlowCtrl(void)
{
   // Set the default flow control settings to enable a connection to a Fasstcom
   // These should be carved in stone somewhere as they took ages to figure out!
   _comPort->FlowControl->OutDSRFlow = false;
   _comPort->FlowControl->DSRSensitivity = false;
   _comPort->FlowControl->ControlDTR = dtrEnable;
   _comPort->FlowControl->ControlRTS = rtsDisable;
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
bool WinCommInterface::isOpen(void)
{
   return (_comPort->Connected);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void WinCommInterface::open(void)
{
   _comPort->Open();
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void WinCommInterface::close(void)
{
   _comPort->Close();
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
int WinCommInterface::setup(void)
{
   if (_owner != 0)
   {
      // Display setup screen for the ComPort
      auto_ptr<TComSetupForm> setupForm (new TComSetupForm(_owner, _comPort, this));

      int result = setupForm->ShowModal();
      if (result == mrOk)
      {
         // Note: The comport library automatically changes the flow control option
         // based on the other flow control settings. The Fasstcom defaults for flow
         // control result in fcNone being set for the Comport component's FlowControl property
         if ((_comPort->FlowControl->FlowControl == fcCustom) || (_comPort->FlowControl->FlowControl == fcNone))
         {
            setDefaultFlowCtrl();
         }
      }
      return (result);
   }
   else
      return (mrCancel);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.: Sets the default com settings for connecting to a WANG/Fasstcom
// Return:
// Notes:
//---------------------------------------------------------------------------
void WinCommInterface::setDefaults()
{
   _comPort->BaudRate = br19200;
   _comPort->DataBits = dbEight;
   _comPort->Parity->Bits = prOdd;
   _comPort->StopBits = sbTwoStopBits;
   _comPort->FlowControl->FlowControl = fcCustom;
   setDefaultFlowCtrl();
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return: true if the setup file was successfully loaded
// Notes:
//---------------------------------------------------------------------------
bool WinCommInterface::loadSetup(const string &fileName)
{
   try
   {
      // Open the file as an ini file
      auto_ptr<TIniFile> setupFile (new TIniFile(fileName.data()));
      // Read the comport, baud rate, data bits, stop bits, parity, flow control
      //### Not the cleanest way to do things
      _comPort->Port = setupFile->ReadString(COM_SECTION, "Port", "COM1");
      _comPort->BaudRate = (TBaudRate)setupFile->ReadInteger(COM_SECTION, "Baud Rate", br19200);
      _comPort->DataBits = (TDataBits)setupFile->ReadInteger(COM_SECTION, "Data Bits", dbEight);
      _comPort->Parity->Bits = (TParityBits)setupFile->ReadInteger(COM_SECTION, "Parity", prOdd);
      _comPort->StopBits = (TStopBits)setupFile->ReadInteger(COM_SECTION, "Stop Bits", sbTwoStopBits);
      TFlowControl tmpFlow = (TFlowControl)setupFile->ReadInteger(COM_SECTION, "Flow Control", fcCustom);
      _comPort->FlowControl->FlowControl = tmpFlow;
      // Note: The comport library automatically changes the flow control option
      // based on the other flow control settings. The Fasstcom defaults for flow
      // control result in fcNone being set for the Comport component's FlowControl property
      if ((tmpFlow == fcCustom) || (tmpFlow == fcNone)) {
         setDefaultFlowCtrl();
      }
      return (true);
   }
   catch (...)
   {
      return (false);
   }
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
bool WinCommInterface::saveSetup(const string &fileName)
{
   try
   {
      // Open the file as an ini file
	  auto_ptr<TIniFile> setupFile (new TIniFile(fileName.data()));
      // Read the comport, baud rate, data bits, stop bits, parity, flow control
      //### Not the cleanest way to do things
      setupFile->WriteString(COM_SECTION, "Port", _comPort->Port);
      setupFile->WriteInteger(COM_SECTION, "Baud Rate", (int)_comPort->BaudRate);
      setupFile->WriteInteger(COM_SECTION, "Data Bits", (int)_comPort->DataBits);
      setupFile->WriteInteger(COM_SECTION, "Parity", (int)_comPort->Parity->Bits);
      setupFile->WriteInteger(COM_SECTION, "Stop Bits", (int)_comPort->StopBits);
      setupFile->WriteInteger(COM_SECTION, "Flow Control", (int)_comPort->FlowControl->FlowControl);
      return (true);
   }
   catch (...)
   {
      return (false);
   }
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return: The number of bytes actually read from the Com port
// Notes:
//---------------------------------------------------------------------------
int WinCommInterface::read(char *buffer, const int count)
{
   return (_comPort->Read(buffer, count));
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return: The number of bytes actually written to the Com port
// Notes:
//---------------------------------------------------------------------------
int WinCommInterface::write(char *buffer, const int count)
{
   return (_comPort->Write(buffer, count));
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void WinCommInterface::getSetupString(char *buffer)
{
   AnsiString baudRate, combined, dataBits, parity, stopBits;

   if (isOpen())
      {
      switch (_comPort->BaudRate)
         {
         case brCustom :
            baudRate = _comPort->CustomBaudRate;
         break;
         case br110 :
            baudRate = "110";
         break;
         case br300 :
            baudRate = "300";
         break;
         case br600 :
            baudRate = "600";
         break;
         case br1200 :
            baudRate = "1200";
         break;
         case br2400 :
            baudRate = "2400";
         break;
         case br4800 :
            baudRate = "4800";
         break;
         case br9600 :
            baudRate = "9600";
         break;
         case br14400 :
            baudRate = "14400";
         break;
         case br19200 :
            baudRate = "19200";
         break;
         case br38400 :
            baudRate = "38400";
         break;
         case br56000 :
            baudRate = "56000";
         break;
         case br57600 :
            baudRate = "57600";
         break;
         case br115200 :
            baudRate = "115200";
         break;
         }
      switch (_comPort->DataBits)
         {
         case dbFive :
            dataBits = "5";
         break;
         case dbSix :
            dataBits = "6";
         break;
         case dbSeven :
            dataBits = "7";
         break;
         case dbEight :
            dataBits = "8";
         break;
         default :
            dataBits = "?";
         break;
         }
      switch (_comPort->Parity->Bits)
         {
         case prEven :
            parity = "E";
         break;
         case prMark :
            parity = "M";
         break;
         case prNone :
            parity = "N";
         break;
         case prOdd :
            parity = "O";
         break;
         case prSpace :
            parity = "S";
         break;
         default :
            parity = "?";
         break;
         }
      switch (_comPort->StopBits)
         {
         case sbOne5StopBits :
            stopBits = "1.5";
         break;
         case sbOneStopBit :
            stopBits = "1";
         break;
         case sbTwoStopBits :
            stopBits = "2";
         break;
         default :
            stopBits = "?";
         break;
         }
      combined = "Connected: " + _comPort->Port + ", " + baudRate + ", " + dataBits + ", " + parity + ", " + stopBits;
      strncpy(buffer, combined.c_str(), combined.Length());
      buffer[combined.Length()] = '\0';
      }
   else
      strcpy(buffer, "Disconnected");
}
//---------------------------------------------------------------------------
// END OF FILE
//---------------------------------------------------------------------------