//---------------------------------------------------------------------------
// Copyright Mark Pickersgill 2001-2014
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// File: ScreenInterface.cpp
// Desc.: Implementation file for the Base ScreenInterface class.
//
//---------------------------------------------------------------------------
// Created: 19/08/2001
// Version History:
//    19/08/2001, 0.0.0 - Created the file
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// Required CUSTOM header files
//---------------------------------------------------------------------------
#include "ScreenInterface.h"
//---------------------------------------------------------------------------
// Required SYSTEM header files
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// --== IMPLEMENTATION ==--
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// Class: ScreenInterface
// Desc.: Base Screen Interface class from which other classes derive
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//  PRIVATE Methods
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// Method: ()
// Desc.: Called by each constructor to initialise variables.
// Return:
// Notes:
//---------------------------------------------------------------------------
void ScreenInterface::initVariables(void)
{
_currentX = 0;
_currentY = 0;
_caretOn = true;
// Initialise the colour array
charAttribs_t defAttribs;
ScreenBuffer::getDefAttribs(defAttribs); // Obtain the set current colours
_defColours[CC_NORMAL_BACK]=defAttribs.BackColour;
_defColours[CC_REVERSE]=IBM_GREEN;
_defColours[CC_NORMAL_FORE]=defAttribs.ForeColour;
_defColours[CC_UNDERLINE]=IBM_GREEN;
_defColours[CC_ALT_FORE1]=IBM_WHITE;
_defColours[CC_REV_UNDERLINE]=IBM_BLACK;
_defColours[CC_ALT_FORE2]=IBM_LIGHTGREEN;
_defColours[CC_BRIGHT]=IBM_WHITE;
_defColours[CC_ALT_FORE3]=IBM_YELLOW;
_defColours[CC_BRIGHT_BLINK]=IBM_LIGHTCYAN;
_defColours[CC_ALT_BACK1]=IBM_RED;
_defColours[CC_ALT_BACK2]=IBM_GREEN;
_defColours[CC_ALT_BACK3]=IBM_BLACK;
_defColours[CC_BOX_LINE]=IBM_YELLOW;
_defColours[CC_BLINK]=IBM_LIGHTGREEN;
_defColours[CC_BLANK]=IBM_BLACK;
defColoursToCur();
_currentAttribs.Attrib1 = ATTR_NORMAL1;
_currentAttribs.Attrib2 = ATTR_NORMAL2;
_storedAttribs = _currentAttribs;
_storedAttribs.Attrib1 = ATTR_BRIGHT;
//FScreenType = SCREEN_WT;
_realReverse = true;
_realUnderline = true;
_realBright = true;
_realBlink = false;
_currentCharSet = CS_NORMAL;
_statusAttribs = _currentAttribs;
_statusX = 0;
//_fontSize = 10; // Arbitry value. Should be set according to the platform specific implementation
// Use the font names for the default resolution in English
calcFontNames(LANG_ENG, 640, 480, ScreenBuffer::getWidth(), ScreenBuffer::getHeight());
}

void ScreenInterface::initScreen()
{
_caretOn = true;
defColoursToCur();
_currentAttribs.Attrib1 = ATTR_NORMAL1;
_currentAttribs.Attrib2 = ATTR_NORMAL2;
_storedAttribs = _currentAttribs;
_storedAttribs.Attrib1 = ATTR_BRIGHT;
_statusAttribs = _currentAttribs;
_currentCharSet = CS_NORMAL;
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.: Checks the current attributes and selects any appropriate
//    colours.
// Return:
// Notes:
//---------------------------------------------------------------------------
void ScreenInterface::chooseCharAttribs(const char character, charAttribs_t &char_attribs)
{
//###Check to see if the Underline will override other attributes
// Check the attributes for the colours to set the character
// attribute to a particular colour
switch (char_attribs.Attrib1)
   {
   case ATTR_BRIGHT :
      // If screen is not set to display bold, then select a colour
      if (!_realBright)
         {
         char_attribs.ForeColour = _curColours[CC_BRIGHT];
         }
      else
         {
         char_attribs.ForeColour = _curColours[CC_NORMAL_FORE];
         if (char_attribs.ForeColour <= IBM_LIGHTGRAY)
            char_attribs.ForeColour =
               (colour_t)(char_attribs.ForeColour + IBM_DARKGRAY);
         }
   break;
   case ATTR_BLINK :
      // If screen is not set to display blinking or italic text, then
      // select a colour
      if (!_realBlink)
         {
         char_attribs.ForeColour = _curColours[CC_BLINK];
         }
      else
         {
         char_attribs.ForeColour = _curColours[CC_NORMAL_FORE];
         }
   break;
   case ATTR_BRIGHT_BLINK :
      // If screen is not set to display blinking or bright text then select
      // a colour
      if (!_realBlink || !_realBright)
         {
         char_attribs.ForeColour = _curColours[CC_BRIGHT_BLINK];
         }
      else
         {
         // Blinking is taken care of in the actual drawing of the characters
         char_attribs.ForeColour = _curColours[CC_NORMAL_FORE];
         if (char_attribs.ForeColour <= IBM_LIGHTGRAY)
            char_attribs.ForeColour =
               (colour_t)(char_attribs.ForeColour + IBM_DARKGRAY);
         }
   break;
   case ATTR_ALT_FORE1 :
      char_attribs.ForeColour = _curColours[CC_ALT_FORE1];
   break;
   case ATTR_ALT_FORE2 :
      char_attribs.ForeColour = _curColours[CC_ALT_FORE2];
   break;
   case ATTR_ALT_FORE3 :
      char_attribs.ForeColour = _curColours[CC_ALT_FORE3];
   break;
   default : //Default to using the Normal foreground colour
      char_attribs.ForeColour = _curColours[CC_NORMAL_FORE];
   break;
   }
switch (char_attribs.Attrib2)
   {
   case ATTR_REVERSE :
      // If the screen is not set to display reverse video then select the
      // reverse video colour
      if (!_realReverse)
         {
         char_attribs.ForeColour = _curColours[CC_REVERSE];
         char_attribs.BackColour = _curColours[CC_NORMAL_BACK];
         }
      else
         {
         // Reverse the colours around. Since ForeColour was set in last
         // switch statement, use it as the Foreground colour to swap
         char_attribs.BackColour = char_attribs.ForeColour;
         char_attribs.ForeColour = _curColours[CC_NORMAL_BACK];
         }
   break;
   case ATTR_UNDERLINE :
      // If screen is not set to display underlined characters then select
      // the underline colour
      if (!_realUnderline)
         {
         char_attribs.ForeColour = _curColours[CC_UNDERLINE];
         }
      char_attribs.BackColour = _curColours[CC_NORMAL_BACK];
   break;
   case ATTR_REV_UNDERLINE :
      // If screen is not set to display reverse or underlined text then select
      // a colour
      if (!_realReverse || !_realUnderline)
         {
         char_attribs.ForeColour = _curColours[CC_REV_UNDERLINE];
         }
      else
         {
         // Underlining is taken care of in the actual drawing of the characters
         char_attribs.BackColour = char_attribs.ForeColour;
         char_attribs.ForeColour = _curColours[CC_NORMAL_BACK];
         }
   break;
   case ATTR_ALT_BACK1 :
      char_attribs.BackColour = _curColours[CC_ALT_BACK1];
   break;
   case ATTR_ALT_BACK2 :
      char_attribs.BackColour = _curColours[CC_ALT_BACK2];
   break;
   case ATTR_ALT_BACK3 :
      char_attribs.BackColour = _curColours[CC_ALT_BACK3];
   break;
   default : // Default to using the normal background colour
      char_attribs.BackColour = _curColours[CC_NORMAL_BACK];
   break;
   }
}

//---------------------------------------------------------------------------
//  PUBLIC Methods
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
ScreenInterface::ScreenInterface()
   : ScreenBuffer(),
   _ficeFont()
{
initVariables();
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
ScreenInterface::ScreenInterface(const int width, const int height)
   : ScreenBuffer(width, height)
{
initVariables();
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
ScreenInterface::ScreenInterface(const int width, const int height, const colour_t fore, const colour_t back)
   : ScreenBuffer(width, height, fore, back)
{
initVariables();
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
inline void ScreenInterface::setCaretOn(const bool state)
{
_caretOn = state;
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
inline bool ScreenInterface::getCaretOn(void) const
{
return (_caretOn);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
bool ScreenInterface::clearScreen(void)
{
//###Need better status line handling
if (_characterArray != 0 && _attributeArray != 0)
   {
   // Only clear the actual drawing screen and not the status line
   int right = getWidth()-1,
      bottom = getHeight()-2;

   fillCharRect(0, 0, right, bottom, DEF_CHAR);
   charAttribs_t char_attribs;
   // Set the correct colours for the characters
   toggleAttribs(false);
   char_attribs.Attrib1 = _currentAttribs.Attrib1;
   char_attribs.Attrib2 = _currentAttribs.Attrib2;
   char_attribs.CharSet = CS_NORMAL;
   char_attribs.Flags = 0;
   chooseCharAttribs(DEF_CHAR, char_attribs);
   fillAttributeRect(0, 0, right, bottom, char_attribs);
   return (true);
   }
else
   {
   return (false);
   }
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
bool ScreenInterface::moveCaret(const direction_t direction, const int count)
{
//###Need better status line handling
//###The wrap variable is for possibly wrapping caret movement
bool wrap = false, result = true;
int height = getHeight()-1,
   width = getWidth(),
   tempcount, temppos;

switch (direction)
   {
   case DIR_UP :
      if (wrap)
         {
         tempcount = count % height;
         temppos = _currentY;
         _currentY = (height+_currentY-tempcount) % height;
         tempcount = ((height-temppos+count-1)/height) % width;
         _currentX = (height+_currentX-tempcount) % width;
         }
      else
         //_currentY = max(0, _currentY-count);
         _currentY = _currentY-count;
      break;
   case DIR_DOWN :
      if (wrap)
         {
         tempcount = count % height;
         temppos = _currentY;
         _currentY = (_currentY+tempcount) % height;
         tempcount = ((temppos+count) / height) % width;
         _currentX = (_currentX+tempcount) % width;
         }
      else
         //_currentY = min(height-1, _currentY+count);
         _currentY = _currentY+count;
      break;
   case DIR_LEFT :
      if (wrap)
         {
         tempcount = count % width;
         temppos = _currentX;
         _currentX = (width+_currentX-tempcount) % width;
         tempcount = ((width-temppos+count-1)/width) % height;
         _currentY = (height+_currentY-tempcount) % height;
         }
      else
         //_currentX = max(0, _currentX-count);
         _currentX = _currentX-count;
      break;
   case DIR_RIGHT :
      if (wrap)
         {
         tempcount = count % width;
         temppos = _currentX;
         _currentX = (_currentX+tempcount) % width;
         tempcount = ((temppos+count) / width) % height;
         _currentY = (_currentY+tempcount) % height;
         }
      else
         //_currentX = min(width-1, _currentX+count);
         // Possibly set an overflow flag to prevent the buffer being written to by mistake
         _currentX = _currentX+count;
      break;
   case DIR_HOME :
      _currentX = 0;
      break;
   case DIR_END :
      _currentX = width-1;
      break;
   case DIR_TOP_HOME :
      _currentX = 0;
      _currentY = 0;
      break;
   case DIR_BOTTOM_END :
      _currentX = width-1;
      _currentY = height-1;
   break;
   default :
      result = false;
      break;
   }
return (result);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void ScreenInterface::setColour(c_control_t c_control, colour_t colour)
{
//Note: in derived classes, a check for c_control==0 needs to be made
//and if it is found, then the screen also needs to be cleared with the
//new colour.
_curColours[c_control] = colour;
}

colour_t ScreenInterface::getColour(c_control_t c_control)
{
return _curColours[c_control];
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void ScreenInterface::setDefColour(c_control_t c_control, colour_t colour)
{
_defColours[c_control] = colour;
}

colour_t ScreenInterface::getDefColour(c_control_t c_control)
{
return _defColours[c_control];
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void ScreenInterface::defColoursToCur()
{
for (int i=0; i < MAX_COLOURS; i++)
   _curColours[i] = _defColours[i];
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
inline void ScreenInterface::setStoredAttribs(const attrib_t &attribute)
{
_storedAttribs = attribute;
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
inline void ScreenInterface::getStoredAttribs(attrib_t &attribute) const
{
attribute = _storedAttribs;
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void ScreenInterface::toggleAttribs(bool on)
{
if (on == true)
   {
   _currentAttribs = _storedAttribs;
   }
else
   {
   _currentAttribs.Attrib1 = ATTR_NORMAL1;
   _currentAttribs.Attrib2 = ATTR_NORMAL2;
   }
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void ScreenInterface::putChar(char character)
{
//Note: derived classes may have to move the cursor right, one position.
charAttribs_t char_attribs = DEF_ATTR;

//###check that the following two assignments are correct!
char_attribs.Attrib1 = _currentAttribs.Attrib1;
char_attribs.Attrib2 = _currentAttribs.Attrib2;
char_attribs.CharSet = _currentCharSet;
// Set the colours for the character based on the attributes
chooseCharAttribs(character, char_attribs);
ScreenBuffer::setCharAt(_currentX, _currentY, character, char_attribs);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void ScreenInterface::scrollDown(int lines)
{
//###Need better status line handling
int height = getHeight()-1;

if (lines < height)
   {
   int width = getWidth(),
      position = lines * width,
      count = (width * height) - position;

   // Scroll the screen up
   memmove(_characterArray, &_characterArray[position], count);
   memmove(_attributeArray, &_attributeArray[position], sizeof(charAttribs_t) * count);
   // Clear the new lines
   position = height - lines;
   fillCharRect(0, position, width-1, height-1, DEF_CHAR);
   // Use the default attributes and colours to clear the new lines
   charAttribs_t char_attribs;
   char_attribs.Attrib1 = ATTR_NORMAL1;
   char_attribs.Attrib2 = ATTR_NORMAL2;
   char_attribs.CharSet = CS_NORMAL;
   char_attribs.Flags = 0;
   chooseCharAttribs(DEF_CHAR, char_attribs);
   fillAttributeRect(0, position, width-1, height-1, char_attribs);
   // Copy the Flags from the status line as they may contain information regarding
   // graphical drawing.
   //###Need better status line handling
   for (int y1=position,y2=getHeight()-1; y1<=position;y1++,y2++)
      {
      for (int x=0; x<width; x++)
         {
         _attributeArray[y1*width+x].Flags = _attributeArray[y2*width+x].Flags;
         // Clear the flags of the status line
         _attributeArray[y2*width+x].Flags = 0;
         }
      }
   }
else if (lines > 0)
   {
   //Since all of the lines in the screen will be scrolled, clear
   //the screen
   clearScreen();
   }
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void ScreenInterface::setCharSet(charSet_t char_set)
{
_currentCharSet = char_set;
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
charSet_t ScreenInterface::getCharSet(void)
{
return (_currentCharSet);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void ScreenInterface::setRealUnderline(bool on)
{
_realUnderline = on;
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void ScreenInterface::setRealReverse(bool on)
{
_realReverse = on;
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void ScreenInterface::setRealBright(bool on)
{
_realBright = on;
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void ScreenInterface::setRealBlink(bool on)
{
_realBlink = on;
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void ScreenInterface::setDrawBoxLines(bool on)
{
_drawBoxLines = on;
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
bool ScreenInterface::getRealUnderline(void)
{
return (_realUnderline);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
bool ScreenInterface::getRealReverse(void)
{
return (_realUnderline);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
bool ScreenInterface::getRealBright(void)
{
return (_realBright);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
bool ScreenInterface::getRealBlink(void)
{
return (_realBlink);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
bool ScreenInterface::getDrawBoxLines()
{
return (_drawBoxLines);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void ScreenInterface::clearStatusLine(void)
{
int left = 0,
   top = getHeight()-1,
   bottom = top,
   right = getWidth()-1;

fillCharRect(left, top, right, bottom, DEF_CHAR);
// DEBUG - Check this for correct attribute
fillAttributeRect(left, top, right, bottom, DEF_ATTR);
_statusAttribs.Attrib1 = ATTR_NORMAL1;
_statusAttribs.Attrib2 = ATTR_NORMAL2;
_statusX = 0;
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void ScreenInterface::setStatusLineAttribs(attrib_t attribs)
{
_statusAttribs = attribs;
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
bool ScreenInterface::writeStatusLine(char character, bool scroll)
{
int width = getWidth(),
   height = getHeight(),
   startpos = (height-1) * width,
   endpos = startpos+1;
//bool result=false;   // Set to true if _statusX has reached the screen width
charAttribs_t char_attribs = DEF_ATTR;

char_attribs.Attrib1 = _statusAttribs.Attrib1;
char_attribs.Attrib2 = _statusAttribs.Attrib2;
char_attribs.CharSet = CS_NORMAL;
// Set the colours for the character based on the attributes
chooseCharAttribs(character, char_attribs);
if (_statusX > width-1)
   {
   //###FT specific screen behaviour
   //###Check this scrolling
   if (scroll)
      {
      //Do a horizontal scroll on the status line
      memmove(&_characterArray[startpos], &_characterArray[endpos], width-1);
      memmove(&_attributeArray[startpos], &_attributeArray[endpos], sizeof(charAttribs_t) * (width-1));
      _characterArray[width-1] = DEF_CHAR;
      _attributeArray[width-1] = char_attribs;
      _statusX--;
      }
   }
// Output the character
ScreenBuffer::setCharAt(_statusX, height-1, character, char_attribs);
_statusX++;
// Check to see if we've reached the screen width
if (_statusX == width)
   return (true);
else
   return (false);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void ScreenInterface::finishStatusLine(void)
{
_statusAttribs.Attrib1 = ATTR_NORMAL1;
_statusAttribs.Attrib2 = ATTR_NORMAL2;
_statusX = 0;
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
language_t ScreenInterface::getLanguage(void)
{
return (_ficeFont.getLanguage());
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
string ScreenInterface::getNormalFontName() const
{
return (_ficeFont.getNormalFontName());
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
string ScreenInterface::getNormalFontName2() const
{
return (_ficeFont.getNormalFontName2());
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
string ScreenInterface::getAlt0FontName(void) const
{
return (_ficeFont.getAlt0FontName());
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
string ScreenInterface::getAlt1FontName(void) const
{
return (_ficeFont.getAlt1FontName());
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
string ScreenInterface::getAlt2FontName(void) const
{
return (_ficeFont.getAlt2FontName());
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes: Names of the fonts used by EMU are platform independant, so we can
//    build the names and store them in here.
//    Only font file extensions, truncation of names or sizes needs to be done
//    in inherited classes.
//---------------------------------------------------------------------------
void ScreenInterface::calcFontNames(language_t language, int resX, int resY,
   int screenX, int screenY)
{
_ficeFont.calcFontNames(language, resX, resY, screenX, screenY);
}

void ScreenInterface::redraw()
{
charAttribs_t attribs;
for (int y=0; y<getHeight(); y++)
   for (int x=0; x<getWidth(); x++)
   {
      getAttributeAt(x, y, attribs);
      chooseCharAttribs(DEF_CHAR, attribs);
      setAttributeAt(x, y, attribs);
   }
}
/*
void ScreenInterface::redrawRect(const int x1, const int y1, const int x2, const int y2)
{
charAttribs_t attribs;
for (int y=y1; y<y2; y++)
   for (int x=x1; x<x2; x++)
   {
      getAttributeAt(x, y, attribs);
      chooseCharAttribs(DEF_CHAR, attribs);
      setAttributeAt(x, y, attribs);
   }
}
/*
//---------------------------------------------------------------------------
//###REMOVED FOR NOW
//  ClearEol
//
//  Description: Clears the text & attributes from the specified position to
//    the end of that line.
//---------------------------------------------------------------------------
/*bool ScreenInterface::clearEol(const int xpos, const int ypos)
{
if (validPosition(xpos, ypos) && (_attributeArray != NULL) &&
   (_characterArray != NULL))
   {
   int clear_size = GetWidth() - xpos;

   FillCharRect(xpos, ypos, clear_size, ypos, DEF_CHAR);
   FillAttributeRect(xpos, ypos, clear_size, ypos, DEF_ATTR);
   return (true);
   }
else
   {
   return (false);
   }
}
*/
//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// END OF FILE
//---------------------------------------------------------------------------