//---------------------------------------------------------------------------
// Copyright Mark Pickersgill 2001-2014
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// File: ScreenBuffer.cpp
// Desc.: Implementation file for the Basic Screen buffer class
//
//---------------------------------------------------------------------------
// Created: 06/07/2001
// Version History:
//    06/07/2001, 0.0.1 - Very basic implementation
//
// SEE HEADER FILE FOR FULL VERSION HISTORY
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// Required CUSTOM header files
//---------------------------------------------------------------------------
#include "ScreenBuffer.h"
//---------------------------------------------------------------------------
// Required SYSTEM header files
//---------------------------------------------------------------------------
#include <string.h>
#include <stdlib.h>
#include <mem.h>
//#include <windef.h>
//---------------------------------------------------------------------------
// --== IMPLEMENTATION ==--
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//  Supporting Functions
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// Method: makeCharAttribs()
// Desc.: Creates a charAttribs_t structure based on sigular items
//  that form a charAttribs_t structure.
// Return:
// Notes:
//---------------------------------------------------------------------------
charAttribs_t makeCharAttribs(colour_t fore, colour_t back, attrib1_t attrib1, attrib2_t attrib2, charSet_t char_set, unsigned char flags)
{
charAttribs_t attributes;
attributes.ForeColour = fore;
attributes.BackColour = back;
attributes.Attrib1 = attrib1;
attributes.Attrib2 = attrib2;
attributes.CharSet = char_set;
attributes.Flags = flags;
return (attributes);
}

//---------------------------------------------------------------------------
// Method: maxPosLast()
// Desc.: Places the highest co-ordinate values FIRST, swapping the
//  values if necessary.
// Return:
// Notes:
//---------------------------------------------------------------------------
void maxPosLast(int &x1, int &y1, int &x2, int &y2)
{
int temp;

if (x1 > x2)
    {
    //Swap the values around
    temp = x1;
    x1 = x2;
    x2 = temp;
    }
if (y1 > y2)
    {
    //Swap the values around
    temp = y1;
    y1 = y2;
    y2 = temp;
    }
}

//---------------------------------------------------------------------------
// Method: minPosLast()
// Desc.: Places the highest co-ordinate values LAST, swapping the
//  values if necessary.
// Return:
// Notes:
//---------------------------------------------------------------------------
void minPosLast(int &x1, int &y1, int &x2, int &y2)
{
int temp;

if (x2 > x1)
    {
    //Swap the values around
    temp = x1;
    x1 = x2;
    x2 = temp;
    }
if (y2 > y1)
    {
    //Swap the values around
    temp = y1;
    y1 = y2;
    y2 = temp;
    }
}
//---------------------------------------------------------------------------
// Class: ScreenBuffer
// Desc.: Basic Screen Class to implement a basic screen buffer for reading
//       and writing to in a similar fashion to a real screen.
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//  PRIVATE Methods
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// Method: initCharArray()
// Desc.: Initialises a char array with the specified value.
// Return:
// Notes: NO BOUNDS CHECKING!
//          THE LENGTH OF THE ARRAY MUST NOT BE EXCEEDED!
//---------------------------------------------------------------------------
void ScreenBuffer::initCharArray(unsigned char *chararray, const unsigned char value, const int count)
{
int i;
for (i=0;i < count; i++, chararray++)
    {
    *chararray = value;
    }
}

//---------------------------------------------------------------------------
// Method: initAttribsArray()
// Desc.: Initialises a charAttribs_t array with the specified value.
// Return:
// Notes: NO BOUNDS CHECKING!
//          THE LENGTH OF THE ARRAY MUST NOT BE EXCEEDED!
//---------------------------------------------------------------------------
void ScreenBuffer::initAttribsArray(charAttribs_t *attrarray, const charAttribs_t value, const int count)
{
int i;
for (i=0;i < count; i++, attrarray++)
    {
    *attrarray = value;
    }
}

//---------------------------------------------------------------------------
// Method: checkForDefault(attributes)
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void ScreenBuffer::checkForDefault(charAttribs_t &attributes) const
{
//Check an attribute for 'default' flags and change the default
//flags to the actual default values
   if (attributes.ForeColour == IBM_DEFAULT_COLOUR)
      attributes.ForeColour = _defaultAttribs.ForeColour;
   if (attributes.BackColour == IBM_DEFAULT_COLOUR)
      attributes.BackColour = _defaultAttribs.BackColour;
   if (attributes.Attrib1 == ATTR_DEFAULT1)
      attributes.Attrib1 = _defaultAttribs.Attrib1;
   if (attributes.Attrib2 == ATTR_DEFAULT2)
      attributes.Attrib2 = _defaultAttribs.Attrib2;
}

//---------------------------------------------------------------------------
// Method: checkForCurrent(x,y,attributes)
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void ScreenBuffer::checkForCurrent(const int xpos, const int ypos, charAttribs_t &attributes) const
{
int position = ypos*_screenWidth+xpos;;
//Check an attribute for 'current' flags and obtain the current attribute value
//at the given co-ordinates and place them into the attribute parameter.
   if (attributes.ForeColour == IBM_CURRENT_COLOUR)
      attributes.ForeColour = _attributeArray[position].ForeColour;
   if (attributes.BackColour == IBM_CURRENT_COLOUR)
      attributes.BackColour = _attributeArray[position].BackColour;
   if (attributes.Attrib1 == ATTR_CURRENT1)
      attributes.Attrib1 = _attributeArray[position].Attrib1;
   if (attributes.Attrib2 == ATTR_CURRENT2)
      attributes.Attrib2 = _attributeArray[position].Attrib2;
}

//---------------------------------------------------------------------------
// Method: initVariables(width, height, fore, back)
// Desc.: Called by constructors to setup the variables
//  If the width and height are not valid values, then the default screen
//  size is taken.
// Return:
// Notes:
//---------------------------------------------------------------------------
void ScreenBuffer::initVariables(const int width, const int height, colour_t fore,
colour_t back)
{
int buff_size=0;

//Check for an invalid screen size
if ((width > SCR_MAX_W) || (width < SCR_TINY_W) ||
    (height > SCR_MAX_H) || (height < SCR_TINY_H) ||
    (height*width > SCR_MAX_BUFFER))
    {
    _screenWidth = SCR_STD_W;
    _screenHeight = SCR_STD_H;
    }
else
   {
   _screenWidth = width;
   _screenHeight = height;
   }
//Setup the Drawing Attributes
//ensure that the Default colours are REAL colours, otherwise
//choose the default colours
if (fore == IBM_DEFAULT_COLOUR || fore == IBM_CURRENT_COLOUR)
   _defaultAttribs.ForeColour = IBM_LIGHTGRAY;
else
   _defaultAttribs.ForeColour = fore;
if (back == IBM_DEFAULT_COLOUR || back == IBM_CURRENT_COLOUR)
   _defaultAttribs.BackColour = IBM_BLACK;
else
   _defaultAttribs.BackColour = back;
_defaultAttribs.Attrib1 = ATTR_NORMAL1;
_defaultAttribs.Attrib2 = ATTR_NORMAL2;
_defaultAttribs.CharSet = CS_NORMAL;
_defaultAttribs.Flags = 0;

//Initialise the arrays
buff_size = _screenWidth*_screenHeight;
_characterArray = new unsigned char[buff_size];
_attributeArray = new charAttribs_t[buff_size];
initCharArray(_characterArray, DEF_CHAR, buff_size);
initAttribsArray(_attributeArray, DEF_ATTR, buff_size);
}
//---------------------------------------------------------------------------
//  PUBLIC Methods
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// Method: ScreenBuffer()
// Desc.: Default Constructor
// Return:
// Notes:
//---------------------------------------------------------------------------
ScreenBuffer::ScreenBuffer()
   : _characterArray(0), _attributeArray(0)
{
initVariables(SCR_STD_W, SCR_STD_H, IBM_DEFAULT_COLOUR, IBM_DEFAULT_COLOUR);
}

//---------------------------------------------------------------------------
// Method: ScreenBuffer(width, height)
// Desc.: Constructor allowing specific screen width and height.
//  If the width and height are not valid values, then the default screen
//  size is taken.
// Return:
// Notes:
//---------------------------------------------------------------------------
ScreenBuffer::ScreenBuffer(int width, int height)
   : _characterArray(0), _attributeArray(0)
{
initVariables(width, height, IBM_DEFAULT_COLOUR, IBM_DEFAULT_COLOUR);
}

//---------------------------------------------------------------------------
// Method: ScreenBuffer(width, height, fore, back)
// Desc.: Constructor allowing specific screen width, height and
//  default colours.
//  If the width and height are not valid values, then the default screen
//  size is taken.
// Return:
// Notes:
//---------------------------------------------------------------------------
ScreenBuffer::ScreenBuffer(int width, int height, colour_t fore, colour_t back)
   : _characterArray(0), _attributeArray(0)
{
initVariables(width, height, fore, back);
}

//---------------------------------------------------------------------------
// Method: ~ScreenBuffer()
// Desc.: Destructor
// Return:
// Notes:
//---------------------------------------------------------------------------
ScreenBuffer::~ScreenBuffer()
{
delete[] _characterArray;
delete[] _attributeArray;
}

//---------------------------------------------------------------------------
// Method: validPosition()
// Desc.: Checks to see if the co-ordinates are within the current
//  screen size.
// Return:
// Notes:
//---------------------------------------------------------------------------
inline bool ScreenBuffer::validPosition(const int xpos, const int ypos) const
{
if (xpos < 0 || xpos >= _screenWidth ||
    ypos < 0 || ypos >= _screenHeight)
    {
    return (false);
    }
else
    return (true);
}

//---------------------------------------------------------------------------
// Method: setDefAttribs()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void ScreenBuffer::setDefAttribs(const charAttribs_t &attributes)
{
charAttribs_t temp_attribs = attributes;

//Firstly check for any current or default flags
if (temp_attribs.ForeColour == IBM_DEFAULT_COLOUR ||
   temp_attribs.ForeColour == IBM_CURRENT_COLOUR)
   temp_attribs.ForeColour = _defaultAttribs.ForeColour;

if (temp_attribs.BackColour == IBM_DEFAULT_COLOUR ||
   temp_attribs.BackColour == IBM_CURRENT_COLOUR)
   temp_attribs.BackColour = _defaultAttribs.BackColour;

if (temp_attribs.Attrib1 == ATTR_DEFAULT1 ||
   temp_attribs.Attrib1 == ATTR_CURRENT1)
   temp_attribs.Attrib1 = _defaultAttribs.Attrib1;

if (temp_attribs.Attrib2 == ATTR_DEFAULT2 ||
   temp_attribs.Attrib2 == ATTR_CURRENT2)
   temp_attribs.Attrib2 = _defaultAttribs.Attrib2;

_defaultAttribs = temp_attribs;
}

//---------------------------------------------------------------------------
// Method: getDefAttribs()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void ScreenBuffer::getDefAttribs(charAttribs_t &attributes) const
{
   attributes = _defaultAttribs;
}

//---------------------------------------------------------------------------
// Method: setWidth()
// Desc.: Changes the width of the screen
// Return:
// Notes:
//---------------------------------------------------------------------------
void ScreenBuffer::setWidth(const int width)
{
int new_buf_size = width*_screenHeight;

if ((width >= SCR_TINY_W) && (width <= SCR_MAX_W) &&
    (new_buf_size <= SCR_MAX_BUFFER))
   {
   int y, minwidth;
   unsigned char *newtext, *newtextline, *oldtextline;
   charAttribs_t *newattr, *newattrline, *oldattrline;

   //Modify the position of the lines within the array
   newtext = new unsigned char[new_buf_size];
   newattr = new charAttribs_t[new_buf_size];
   //Initialize the new arrays
   initCharArray(newtext, DEF_CHAR, new_buf_size);
   initAttribsArray(newattr, DEF_ATTR, new_buf_size);
   //Set the minimum screen width to either the old or new line length
   // and copy that amount from the old screen buffer
   minwidth = min(width,_screenWidth);
   for (y=0;y<_screenHeight;y++)
       {
       //Move each line from the old text array to the new one
       oldtextline = &_characterArray[_screenWidth*y];
       newtextline = &newtext[width*y];
       memcpy(newtextline, oldtextline, minwidth*sizeof(unsigned char));
       //Move each line from the old attribute array to the new one
       oldattrline = &_attributeArray[_screenWidth*y];
       newattrline = &newattr[width*y];
       memcpy(newattrline, oldattrline, minwidth*sizeof(charAttribs_t));
       }
   //Delete the current text and attribute buffers
   delete[] _characterArray;
   delete[] _attributeArray;
   //Use the new buffers as the current text and attribute buffers
   _characterArray = newtext;
   _attributeArray = newattr;
   newtext = 0;
   newattr = 0;
   _screenWidth = width;
   }
}

//---------------------------------------------------------------------------
// Method: setHeight()
// Desc.: Changes the height of the screen
// Return:
// Notes:
//---------------------------------------------------------------------------
void ScreenBuffer::setHeight(const int height)
{
int new_buf_size = height*_screenWidth;

if ((height >= SCR_TINY_H) && (height <= SCR_MAX_H) &&
    (new_buf_size <= SCR_MAX_BUFFER))
   {
   int minsize;
   unsigned char *newtext;
   charAttribs_t *newattr;

   //Modify the position of the lines within the array
   newtext = new unsigned char[new_buf_size];
   newattr = new charAttribs_t[new_buf_size];
   //Initialize the new array
   initCharArray(newtext, DEF_CHAR, new_buf_size);
   initAttribsArray(newattr, DEF_ATTR, new_buf_size);
   //Minimum buffer size of new screen and old screen
   minsize = min(height, _screenHeight) * _screenWidth;
   //Move the text array
   memcpy(newtext, _characterArray, minsize * sizeof(unsigned char));
   //Move the attribute array
   memcpy(newattr, _attributeArray, minsize * sizeof(charAttribs_t));
   //Delete the current text and attribute buffers
   delete[] _characterArray;
   delete[] _attributeArray;
   //Use the new buffers as the current text and attribute buffers
   _characterArray = newtext;
   _attributeArray = newattr;
   newtext = 0;
   newattr = 0;
   _screenHeight = height;
   }
}

//---------------------------------------------------------------------------
// Method: getWidth()
// Desc.:
// Return: Returns the current width of the screen
// Notes:
//---------------------------------------------------------------------------
inline int ScreenBuffer::getWidth(void) const
{
return (_screenWidth);
}

//---------------------------------------------------------------------------
// Method: getHeight()
// Desc.:
// Return: Returns the current height of the screen
// Notes:
//---------------------------------------------------------------------------
inline int ScreenBuffer::getHeight(void) const
{
return (_screenHeight);
}

//---------------------------------------------------------------------------
// Method: setCharAt()
// Desc.: Sets the character at the specified x,y position.
// Return:
// Notes:
//---------------------------------------------------------------------------
bool ScreenBuffer::setCharAt(const int xpos, const int ypos, const unsigned char character)
{
if (validPosition(xpos, ypos) && _characterArray != 0)
   {
   _characterArray[ypos*_screenWidth+xpos] = character;
   return (true);
   }
else
   {
   return (false);
   }
}

//---------------------------------------------------------------------------
// Method: setCharAt()
// Desc.: Sets the character at the specified x,y position as well as
//    the attributes at that position.
// Return:
// Notes: The Flags property of the supplied attributes are ignored.
//---------------------------------------------------------------------------
bool ScreenBuffer::setCharAt(const int xpos, const int ypos, const unsigned char character, const charAttribs_t &attributes)
{
if ((_attributeArray != 0) && setCharAt(xpos, ypos, character))
   {
   int position = ypos*_screenWidth+xpos;
   charAttribs_t temp_attribs = attributes;

   temp_attribs.Flags = _attributeArray[position].Flags;
   checkForCurrent(xpos, ypos, temp_attribs);
   _attributeArray[position] = temp_attribs;
   return (true);
   }
else
   {
   return (false);
   }
}

//---------------------------------------------------------------------------
// Method: getCharAt()
// Desc.:
// Return: Returns the character in the specified x,y co-ordinate.
// Notes:
//---------------------------------------------------------------------------
unsigned char ScreenBuffer::getCharAt(const int xpos, const int ypos) const
{
if (validPosition(xpos, ypos) && _characterArray != 0)
   {
   return (_characterArray[ypos*_screenWidth+xpos]);
   }
else
   {
   //Default to returning a null character
   return ('\0');
   }
}

//---------------------------------------------------------------------------
// Method: getCharAt()
// Desc.:
// Return: Returns the character in the specified x,y co-ordinate,
//    as well as the attribute at the specified position.
// Notes:
//---------------------------------------------------------------------------
unsigned char ScreenBuffer::getCharAt(const int xpos, const int ypos, charAttribs_t &attributes) const
{
unsigned char char_at;

if (_attributeArray != 0)
   {
   char_at = getCharAt(xpos, ypos);
   attributes = _attributeArray[ypos*_screenWidth+xpos];
   checkForDefault(attributes);
   return (char_at);
   }
else
   {
   //Default to returning a null character
   return ('\0');
   }
}

//---------------------------------------------------------------------------
// Method: fillCharRect()
// Desc.: Fills the specified rectangle (inclusive) of text with the
//    specified character.
// Return:
// Notes:
//---------------------------------------------------------------------------
bool ScreenBuffer::fillCharRect(const int x1, const int y1, const int x2, const int y2, const unsigned char character)
{
if (validPosition(x1, y1) && validPosition(x2, y2) &&
   (_characterArray != 0))
   {
   for (int ypos=y1; ypos <= y2; ypos++)
      for (int xpos=x1; xpos <= x2; xpos++)
         {
         _characterArray[ypos*_screenWidth+xpos] = character;
         }
   return (true);
   }
else
   {
   return (false);
   }
}

bool ScreenBuffer::setFlagsAt(const int xpos, const int ypos, const unsigned char flags)
{
if (validPosition(xpos, ypos) && _attributeArray != 0)
   {
   int position = ypos*_screenWidth+xpos;
   _attributeArray[position].Flags = flags;
   return (true);
   }
else
   {
   return (false);
   }
}

bool ScreenBuffer::getFlagsAt(const int xpos, const int ypos, unsigned char &flags) const
{
if (validPosition(xpos, ypos) && _attributeArray != 0)
   {
   int position = ypos*_screenWidth+xpos;
   flags = _attributeArray[position].Flags;
   return (true);
   }
else
   {
   return (false);
   }
}

//---------------------------------------------------------------------------
// Method: setAttributeAt()
// Desc.:
// Return:
// Notes: The Flags property of the supplied attributes is ignored
//---------------------------------------------------------------------------
bool ScreenBuffer::setAttributeAt(const int xpos, const int ypos, const charAttribs_t &attributes)
{
if (validPosition(xpos, ypos) && _attributeArray != 0)
   {
   int position = ypos*_screenWidth+xpos;
   charAttribs_t temp_attribs = attributes;

   temp_attribs.Flags = _attributeArray[position].Flags;
   checkForCurrent(xpos, ypos, temp_attribs);
   _attributeArray[position] = temp_attribs;
   return (true);
   }
else
   {
   return (false);
   }
}

//---------------------------------------------------------------------------
// Method: getAttributeAt()
// Desc.:
// Return: Returns the attribute in the specified x,y position.
// Notes:
//---------------------------------------------------------------------------
bool ScreenBuffer::getAttributeAt(const int xpos, const int ypos, charAttribs_t &attributes) const
{
if (validPosition(xpos, ypos) && _attributeArray != 0)
   {
   attributes = _attributeArray[ypos*_screenWidth+xpos];
   checkForDefault(attributes);
   return (true);
   }
else
   {
   return (false);
   }
}

//---------------------------------------------------------------------------
// Method: fillAttributeRect()
// Desc.: Fills the specified rectangle (inclusive) of attributes with
//    the specified attribute.
// Return:
// Notes:
//---------------------------------------------------------------------------
bool ScreenBuffer::fillAttributeRect(const int x1, const int y1, const int x2, const int y2, const charAttribs_t &attribute)
{
if (validPosition(x1, y1) && validPosition(x2, y2) &&
   (_attributeArray != 0))
   {
   for (int ypos=y1; ypos <= y2; ypos++)
      for (int xpos=x1; xpos <= x2; xpos++)
         {
         _attributeArray[ypos*_screenWidth+xpos] = attribute;
         }
   return (true);
   }
else
   {
   return (false);
   }
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// END OF FILE
//---------------------------------------------------------------------------
