//---------------------------------------------------------------------------
// Copyright Mark Pickersgill 2001-2014
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
#ifndef KeyMapH
#define KeyMapH
//---------------------------------------------------------------------------
// File: KeyMap.h
// Desc.: Declaration file for the KeyMap base class
//
//###Rename to KeyboardMap
//---------------------------------------------------------------------------
// Created: 19/08/2001
// Version History:
//    19/08/2001, 0.0.0 - First, skeleton interface declaration.
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// Required CUSTOM header files
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// Required SYSTEM header files
//---------------------------------------------------------------------------
#include <fstream.h>
#include <string.h>
#include <string>
#include "strcommon.h"
//---------------------------------------------------------------------------
// --== Macros and constants ==--
//---------------------------------------------------------------------------
//#define MAP_SERIES      256
#define MAP_SIZE        1024
#define PREMAP_SIZE     73
#define MAX_PNAME       10
#define MAX_PCODE       2
#define MAX_CODE        256
#define MAX_NAME        20
#define KEYMAP_HEADER   "[EMU_KEYMAP]"
#define HEADER_LENGTH   13
#define HEX_DELIMETER   'x'
#define STR_DELIMETER   '\"'
#define PRE_DELIMETER   'p'
//---------------------------------------------------------------------------
// --== DECLARATIONS ==--
//---------------------------------------------------------------------------
struct shiftState_t
{
bool ctrl;     // The Ctrl key was held down
bool alt;      // The Alt key was held down
bool shift;    // The Shift key was held down
bool left;     // The Left mouse button was held down
bool right;    // The Right mouse button was held down
bool middle;   // The Middle mouse button was held down
bool dClicked; // The Mouse button was double clicked
};

// Type for a pre-defined key
struct preMap_t
{
char name[MAX_PNAME+1];  // Name of pre-defined key
unsigned char code[MAX_PCODE];   // 2 byte String for key
int codeLength;         // Number of characters in the code buffer
};

// The possible key code types
enum type_t { MT_PREDEFINED, MT_STRING, MT_HEX };

// The following structure defines a single key mapping
struct keyMap
{
char name[MAX_NAME+1];
unsigned char *code;
type_t type;
int preMapIndex;
int codeLength;
};

enum default_t { DT_EMU, DT_BANK1, DT_BANK2, DT_PC, DT_NIAKWA };

enum controlConstant
{
C_UNSHIFT = 0x0000,
C_SHIFT = 0x0100,
C_CTRL = 0x0200,
C_ALT = 0x0300
};

enum keyConstants
{
K_BACK=0x08,
K_TAB=0x09,
K_CLEAR=0x0C,   //###Unsure if this key is needed
K_RETURN=0x0D,
K_ESCAPE=0x1B,
K_SPACE=0x20,
K_PRIOR=0x21,
K_NEXT=0x22,
K_END=0x23,
K_HOME=0x24,
K_LEFT=0x25,
K_UP=0x26,
K_RIGHT=0x27,
K_DOWN=0x28,
K_SELECT=0x29,  //###Unsure if this key is needed
K_EXECUTE=0x2B, //###Unsure if this key is needed
K_INSERT=0x2D,
K_DELETE=0x2E,
K_0=0x30,
K_1=0x31,
K_2=0x32,
K_3=0x33,
K_4=0x34,
K_5=0x35,
K_6=0x36,
K_7=0x37,
K_8=0x38,
K_9=0x39,
K_A=0x41,
K_B=0x42,
K_C=0x43,
K_D=0x44,
K_E=0x45,
K_F=0x46,
K_G=0x47,
K_H=0x48,
K_I=0x49,
K_J=0x4A,
K_K=0x4B,
K_L=0x4C,
K_M=0x4D,
K_N=0x4E,
K_O=0x4F,
K_P=0x50,
K_Q=0x51,
K_R=0x52,
K_S=0x53,
K_T=0x54,
K_U=0x55,
K_V=0x56,
K_W=0x57,
K_X=0x58,
K_Y=0x59,
K_Z=0x5A,
K_NUMPAD0=0x60,
K_NUMPAD1=0x61,
K_NUMPAD2=0x62,
K_NUMPAD3=0x63,
K_NUMPAD4=0x64,
K_NUMPAD5=0x65,
K_NUMPAD6=0x66,
K_NUMPAD7=0x67,
K_NUMPAD8=0x68,
K_NUMPAD9=0x69,
K_MULTIPLY=0x6A,
K_ADD=0x6B,
K_SEPARATOR=0x6C,  //###Unsure if this key is needed
K_SUBTRACT=0x6D,
K_DECIMAL=0x6E,
K_DIVIDE=0x6F,
K_F1=0x70,
K_F2=0x71,
K_F3=0x72,
K_F4=0x73,
K_F5=0x74,
K_F6=0x75,
K_F7=0x76,
K_F8=0x77,
K_F9=0x78,
K_F10=0x79,
K_F11=0x7A,
K_F12=0x7B,
//###Note, the following codes MAY vary on other machines (different keyboards)
K_SEMICOLON=0xBA,
K_EQUALS=0xBB,
K_COMMA=0xBC,
K_MINUS=0xBD,
K_FULLSTOP=0xBE,
K_FORWARDSLASH=0xBF,
K_INVQUOTE=0xC0,
K_SQUARELEFT=0xDB,
K_BACKSLASH=0xDC,
K_SQUARERIGHT=0xDD,
K_SINGLEQUOTE=0xDE
};

//---------------------------------------------------------------------------
// --== Supporting Functions ==--
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// Class: KeyMap
// Desc.: KeyMap provides the necessary methods for loading a keyboard
//    mapping table, editing a mapping table, and translating an input
//    key to a key or string that is in the mapping table
//---------------------------------------------------------------------------
class KeyMap
{
private :
   keyMap **_mapTable;
   string *_fileName;
   void setKeyNames(void);
   void loadBasicKeys(void);
   void loadPcKeys(void);
   void loadBank1Keys(void);
   void loadBank2Keys(void);
   void loadNiakwaKeys(void);
   void loadEmuKeys(void);
public :
   KeyMap();
   ~KeyMap();
   int getPreMapIndex(const char *name);
   void getPreMapName(const int index, char *name);
   void getPreMapCode(const int index, unsigned char *code, int &length);
   void getPreMapCode(const char *name, unsigned char *code, int &length);
   void getKeyName(const unsigned int key, const shiftState_t sState, char *name);
   void getKeyCode(const unsigned int key, const shiftState_t sState, unsigned char *code, int &length);
   type_t getKeyType(const unsigned int key, const shiftState_t sState);
   int getKeyPreMapIndex(const unsigned int key, const shiftState_t sState);
   void assignKey(const unsigned int key,
      unsigned char *code,
      type_t type,
      int preMapIndex = -1);
   void loadDefault(default_t defaultType=DT_EMU);
   bool loadSetup(const string &fileName);
   bool saveSetup(const string &fileName);
   void getFileName(string &fileName);
   virtual void setup(void)=0;
};

#endif

/*
K_LBUTTON='\x01', Left mouse button
K_RBUTTON	02	Right mouse button
K_CANCEL	03	Control-break processing
K_MBUTTON	04	Middle mouse button (three-button mouse)
//  	05-07	Undefined
K_BACK	08	BACKSPACE key
K_TAB	09	TAB key
  	0A-0B	Undefined
K_CLEAR	0C	CLEAR key
K_RETURN	0D	ENTER key
  	0E-0F	Undefined
K_SHIFT	10	SHIFT key
K_CONTROL	11	CTRL key
K_MENU	12	ALT key
K_PAUSE	13	PAUSE key
K_CAPITAL	14	CAPS LOCK key
  	15-19	Reserved for Kanji systems
  	1A	Undefined
K_ESCAPE	1B	ESC key
  	1C-1F	Reserved for Kanji systems
K_SPACE	20	SPACEBAR
K_PRIOR	21	PAGE UP key
K_NEXT	22	PAGE DOWN key
K_END	23	END key
K_HOME	24	HOME key
K_LEFT	25	LEFT ARROW key
K_UP	26	UP ARROW key
K_RIGHT	27	RIGHT ARROW key
K_DOWN	28	DOWN ARROW key
K_SELECT	29	SELECT key
  	2A	Original equipment manufacturer (OEM) specific
K_EXECUTE	2B	EXECUTE key
K_SNAPSHOT	2C	PRINT SCREEN key for Windows 3.0 and later
K_INSERT	2D	INS key
K_DELETE	2E	DEL key
K_HELP	2F	HELP key
K_0	30	0 key
K_1	31	1 key
K_2	32	2 key
K_3	33	3 key
K_4	34	4 key
K_5	35	5 key
K_6	36	6 key
K_7	37	7 key
K_8	38	8 key
K_9	39	9 key
  	3A-40	Undefined
K_A	41	A key
K_B	42	B key
K_C	43	C key
K_D	44	D key
K_E	45	E key
K_F	46	F key
K_G	47	G key
K_H	48	H key
K_I	49	I key
K_J	4A	J key
K_K	4B	K key
K_L	4C	L key
K_M	4D	M key
K_N	4E	N key
K_O	4F	O key
K_P	50	P key
K_Q	51	Q key
K_R	52	R key
K_S	53	S key
K_T	54	T key
K_U	55	U key
K_V	56	V key
K_W	57	W key
K_X	58	X key
K_Y	59	Y key
K_Z	5A	Z key
K_LWIN	5B	Left Windows key (Microsoft Natural Keyboard)
K_RWIN	5C	Right Windows key (Microsoft Natural Keyboard)
K_APPS	5D	Applications key (Microsoft Natural Keyboard)
  	5E-5F	Undefined
K_NUMPAD0	60	Numeric keypad 0 key
K_NUMPAD1	61	Numeric keypad 1 key
K_NUMPAD2	62	Numeric keypad 2 key
K_NUMPAD3	63	Numeric keypad 3 key
K_NUMPAD4	64	Numeric keypad 4 key
K_NUMPAD5	65	Numeric keypad 5 key
K_NUMPAD6	66	Numeric keypad 6 key
K_NUMPAD7	67	Numeric keypad 7 key
K_NUMPAD8	68	Numeric keypad 8 key
K_NUMPAD9	69	Numeric keypad 9 key
K_MULTIPLY	6A	Multiply key
K_ADD	6B	Add key
K_SEPARATOR	6C	Separator key
K_SUBTRACT	6D	Subtract key
K_DECIMAL	6E	Decimal key
K_DIVIDE	6F	Divide key
K_F1	70	F1 key
K_F2	71	F2 key
K_F3	72	F3 key
K_F4	73	F4 key
K_F5	74	F5 key
K_F6	75	F6 key
K_F7	76	F7 key
K_F8	77	F8 key
K_F9	78	F9 key
K_F10	79	F10 key
K_F11	7A	F11 key
K_F12	7B	F12 key
K_F13	7C	F13 key
K_F14	7D	F14 key
K_F15	7E	F15 key
K_F16	7F	F16 key
K_F17	80H	F17 key
K_F18	81H	F18 key
K_F19	82H	F19 key
K_F20	83H	F20 key
K_F21	84H	F21 key
K_F22	85H	F22 key
K_F23	86H	F23 key
K_F24	87H	F24 key
  	88-8F	Unassigned
K_NUMLOCK	90	NUM LOCK key
K_SCROLL	91	SCROLL LOCK key
  	92-B9	Unassigned
  	BA-C0	OEM specific
  	C1-DA	Unassigned
  	DB-E4	OEM specific
  	E5	Unassigned
  	E6	OEM specific
  	E7-E8	Unassigned
  	E9-F5	OEM specific
K_ATTN	F6	Attn key
K_CRSEL	F7	CrSel key
K_EXSEL	F8	ExSel key
K_EREOF	F9	Erase EOF key
K_PLAY	FA	Play key
K_ZOOM	FB	Zoom key
K_NONAME	FC	Reserved for future use.
K_PA1	FD	PA1 key
K_OEM_CLEAR	FE	Clear key
*/

//---------------------------------------------------------------------------
// END OF FILE
//---------------------------------------------------------------------------