//---------------------------------------------------------------------------
// Copyright Mark Pickersgill 2001-2014
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// File: KeyMap.cpp
// Desc.: Implementation file for the KeyMap class
//
//---------------------------------------------------------------------------
// Created: 19/08/2001
// Version History:
//    19/08/2001, 0.0.0 - First skeleton Implementation
//
// SEE HEADER FILE FOR FULL VERSION HISTORY
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// Required CUSTOM header files
//---------------------------------------------------------------------------
#include "KeyMap.h"
//---------------------------------------------------------------------------
// Required SYSTEM header files
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// --== IMPLEMENTATION ==--
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// Supporting Functions
//---------------------------------------------------------------------------
preMap_t preMap[PREMAP_SIZE] = {
{ "NONE",      "\x00", 1 },
{ "SF0",       "\xFD\x00", 2 },
{ "SF1",       "\xFD\x01", 2 },
{ "SF2",       "\xFD\x02", 2 },
{ "SF3",       "\xFD\x03", 2 },
{ "SF4",       "\xFD\x04", 2 },
{ "SF5",       "\xFD\x05", 2 },
{ "SF6",       "\xFD\x06", 2 },
{ "SF7",       "\xFD\x07", 2 },
{ "SF8",       "\xFD\x08", 2 },
{ "SF9",       "\xFD\x09", 2 },
{ "SF10",      "\xFD\x0A", 2 },
{ "SF11",      "\xFD\x0B", 2 },
{ "SF12",      "\xFD\x0C", 2 },
{ "SF13",      "\xFD\x0D", 2 },
{ "SF14",      "\xFD\x0E", 2 },
{ "SF15",      "\xFD\x0F", 2 },
{ "SF16",      "\xFD\x10", 2 },
{ "SF17",      "\xFD\x11", 2 },
{ "SF18",      "\xFD\x12", 2 },
{ "SF19",      "\xFD\x13", 2 },
{ "SF20",      "\xFD\x14", 2 },
{ "SF21",      "\xFD\x15", 2 },
{ "SF22",      "\xFD\x16", 2 },
{ "SF23",      "\xFD\x17", 2 },
{ "SF24",      "\xFD\x18", 2 },
{ "SF25",      "\xFD\x19", 2 },
{ "SF26",      "\xFD\x1A", 2 },
{ "SF27",      "\xFD\x1B", 2 },
{ "SF28",      "\xFD\x1C", 2 },
{ "SF29",      "\xFD\x1D", 2 },
{ "SF30",      "\xFD\x1E", 2 },
{ "SF31",      "\xFD\x1F", 2 },
{ "RESET",     "\x12", 1 },
{ "HALT",      "\x13", 1 },
{ "EXECUTE",   "\xFD\x82", 2 },     //Verify
{ "SHERASE",   "\xE5", 1 },
{ "ACTION",    "\xFD\x01", 2 },
{ "CANCEL",    "\xFD\x0F", 2 },     // Was set to E1, appears as though it should be 0F instead
{ "RECALL",    "\xFD\x4F", 2 },
{ "HOME",      "\xFD\x47", 2 },     //Check Use
{ "PREV",      "\xFD\x42", 2 },
{ "NEXT",      "\xFD\x43", 2 },
{ "SHHOME",    "\xFD\x44", 2 },     //Check Use
{ "DOWN",      "\xFD\x45", 2 },
{ "UP",        "\xFD\x46", 2 },
{ "ERASE",     "\xFD\x48", 2 },
{ "DELETE",    "\xFD\x49", 2 },
{ "INSERT",    "\xFD\x4A", 2 },
{ "RIGHT",     "\xFD\x4C", 2 },
{ "LEFT",      "\xFD\x4D", 2 },
{ "SHPREV",    "\xFD\x52", 2 },
{ "SHNEXT",    "\xFD\x53", 2 },
{ "SHDOWN",    "\xFD\x55", 2 },
{ "SHUP",      "\xFD\x56", 2 },
{ "SHDELETE",  "\xFD\x59", 2 },
{ "TAB",       "\xFD\x7E", 2 },
{ "FN",        "", 0 },          //???###
{ "SHTAB",     "\xFD\x7F", 2 },
{ "SHCANCEL",  "\xFD\x50", 2 },  //Verify
{ "SHINSERT",  "\xFD\x5A", 2 },  //Verify
{ "SHRIGHT",   "\xFD\x5C", 2 },
{ "SHLEFT",    "\xFD\x5D", 2 },
{ "DECTAB",    "\xFD\x60", 2 },  //Diff: 5F
{ "SHFN",      "", 0 },          //???###
{ "GL",        "\xFD\x7C", 2 },
{ "SHGL",      "\xFD\x7D", 2 },
{ "CLEAR",     "\xFD\x81", 2 },
{ "RUN",       "\xFD\x82", 2 },
{ "CONTINUE",  "\xFD\x84", 2 },
{ "ULINE",     "\xFD\xA0", 2 },
{ "LOAD",      "\xFD\xA1", 2 },
{ "EDIT",      "\xBD", 1 }   
};

//---------------------------------------------------------------------------
// Class: KeyMap
// Desc.: KeyMap provides the necessary methods for loading a keyboard
//    mapping table, editing a mapping table, and translating an input
//    key to a key or string that is in the mapping table
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// PRIVATE Methods
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void KeyMap::loadBasicKeys(void)
{
assignKey(K_TAB, "TAB", MT_PREDEFINED);
assignKey(K_PRIOR, "PREV", MT_PREDEFINED);
assignKey(K_NEXT, "NEXT", MT_PREDEFINED);
assignKey(K_INSERT, "INSERT", MT_PREDEFINED);
assignKey(K_DELETE, "DELETE", MT_PREDEFINED);
assignKey(K_LEFT, "LEFT", MT_PREDEFINED);
assignKey(K_UP, "UP", MT_PREDEFINED);
assignKey(K_DOWN, "DOWN", MT_PREDEFINED);
assignKey(K_RIGHT, "RIGHT", MT_PREDEFINED);
assignKey(K_BACK, "08", MT_HEX);
assignKey(K_RETURN, "0D", MT_HEX);
//assignKey(K_ESCAPE, "00", MT_HEX);
assignKey(K_SPACE, " ", MT_STRING);  
assignKey(K_0, "0", MT_STRING);
assignKey(K_1, "1", MT_STRING);
assignKey(K_2, "2", MT_STRING);
assignKey(K_3, "3", MT_STRING);
assignKey(K_4, "4", MT_STRING);
assignKey(K_5, "5", MT_STRING);
assignKey(K_6, "6", MT_STRING);
assignKey(K_7, "7", MT_STRING);
assignKey(K_8, "8", MT_STRING);
assignKey(K_9, "9", MT_STRING);
assignKey(K_A, "a", MT_STRING);
assignKey(K_B, "b", MT_STRING);
assignKey(K_C, "c", MT_STRING);
assignKey(K_D, "d", MT_STRING);
assignKey(K_E, "e", MT_STRING);
assignKey(K_F, "f", MT_STRING);
assignKey(K_G, "g", MT_STRING);
assignKey(K_H, "h", MT_STRING);
assignKey(K_I, "i", MT_STRING);
assignKey(K_J, "j", MT_STRING);
assignKey(K_K, "k", MT_STRING);
assignKey(K_L, "l", MT_STRING);
assignKey(K_M, "m", MT_STRING);
assignKey(K_N, "n", MT_STRING);
assignKey(K_O, "o", MT_STRING);
assignKey(K_P, "p", MT_STRING);
assignKey(K_Q, "q", MT_STRING);
assignKey(K_R, "r", MT_STRING);
assignKey(K_S, "s", MT_STRING);
assignKey(K_T, "t", MT_STRING);
assignKey(K_U, "u", MT_STRING);
assignKey(K_V, "v", MT_STRING);
assignKey(K_W, "w", MT_STRING);
assignKey(K_X, "x", MT_STRING);
assignKey(K_Y, "y", MT_STRING);
assignKey(K_Z, "z", MT_STRING);
assignKey(K_NUMPAD0, "0", MT_STRING);
assignKey(K_NUMPAD1, "1", MT_STRING);
assignKey(K_NUMPAD2, "2", MT_STRING);
assignKey(K_NUMPAD3, "3", MT_STRING);
assignKey(K_NUMPAD4, "4", MT_STRING);
assignKey(K_NUMPAD5, "5", MT_STRING);
assignKey(K_NUMPAD6, "6", MT_STRING);
assignKey(K_NUMPAD7, "7", MT_STRING);
assignKey(K_NUMPAD8, "8", MT_STRING);
assignKey(K_NUMPAD9, "9", MT_STRING);
assignKey(K_MULTIPLY, "*", MT_STRING);
assignKey(K_ADD, "+", MT_STRING);
//_mapTable[K_SEPARATOR] = "";
assignKey(K_SUBTRACT, "-", MT_STRING);
assignKey(K_DECIMAL, ".", MT_STRING);
assignKey(K_DIVIDE, "/", MT_STRING);
assignKey(K_INVQUOTE, "`", MT_STRING);
assignKey(K_MINUS, "-", MT_STRING);
assignKey(K_EQUALS, "=", MT_STRING);
assignKey(K_BACKSLASH, "\\", MT_STRING);
assignKey(K_SQUARELEFT, "[", MT_STRING);
assignKey(K_SQUARERIGHT, "]", MT_STRING);
assignKey(K_SEMICOLON, ";", MT_STRING);
assignKey(K_SINGLEQUOTE, "\'", MT_STRING);
assignKey(K_COMMA, ",", MT_STRING);
assignKey(K_FULLSTOP, ".", MT_STRING);
assignKey(K_FORWARDSLASH, "/", MT_STRING);

// Initialize the Shifted map
assignKey(C_SHIFT | K_0, ")", MT_STRING);
assignKey(C_SHIFT | K_1, "!", MT_STRING);
assignKey(C_SHIFT | K_2, "@", MT_STRING);
assignKey(C_SHIFT | K_3, "#", MT_STRING);
assignKey(C_SHIFT | K_4, "$", MT_STRING);
assignKey(C_SHIFT | K_5, "%", MT_STRING);
assignKey(C_SHIFT | K_6, "^", MT_STRING);
assignKey(C_SHIFT | K_7, "&", MT_STRING);
assignKey(C_SHIFT | K_8, "*", MT_STRING);
assignKey(C_SHIFT | K_9, "(", MT_STRING);
assignKey(C_SHIFT | K_A, "A", MT_STRING);
assignKey(C_SHIFT | K_B, "B", MT_STRING);
assignKey(C_SHIFT | K_C, "C", MT_STRING);
assignKey(C_SHIFT | K_D, "D", MT_STRING);
assignKey(C_SHIFT | K_E, "E", MT_STRING);
assignKey(C_SHIFT | K_F, "F", MT_STRING);
assignKey(C_SHIFT | K_G, "G", MT_STRING);
assignKey(C_SHIFT | K_H, "H", MT_STRING);
assignKey(C_SHIFT | K_I, "I", MT_STRING);
assignKey(C_SHIFT | K_J, "J", MT_STRING);
assignKey(C_SHIFT | K_K, "K", MT_STRING);
assignKey(C_SHIFT | K_L, "L", MT_STRING);
assignKey(C_SHIFT | K_M, "M", MT_STRING);
assignKey(C_SHIFT | K_N, "N", MT_STRING);
assignKey(C_SHIFT | K_O, "O", MT_STRING);
assignKey(C_SHIFT | K_P, "P", MT_STRING);
assignKey(C_SHIFT | K_Q, "Q", MT_STRING);
assignKey(C_SHIFT | K_R, "R", MT_STRING);
assignKey(C_SHIFT | K_S, "S", MT_STRING);
assignKey(C_SHIFT | K_T, "T", MT_STRING);
assignKey(C_SHIFT | K_U, "U", MT_STRING);
assignKey(C_SHIFT | K_V, "V", MT_STRING);
assignKey(C_SHIFT | K_W, "W", MT_STRING);
assignKey(C_SHIFT | K_X, "X", MT_STRING);
assignKey(C_SHIFT | K_Y, "Y", MT_STRING);
assignKey(C_SHIFT | K_Z, "Z", MT_STRING);
assignKey(C_SHIFT | K_INVQUOTE, "~", MT_STRING);
assignKey(C_SHIFT | K_MINUS, "_", MT_STRING);
assignKey(C_SHIFT | K_EQUALS, "+", MT_STRING);
assignKey(C_SHIFT | K_BACKSLASH, "|", MT_STRING);
assignKey(C_SHIFT | K_SQUARELEFT, "{", MT_STRING);
assignKey(C_SHIFT | K_SQUARERIGHT, "}", MT_STRING);
assignKey(C_SHIFT | K_SEMICOLON, ":", MT_STRING);
assignKey(C_SHIFT | K_SINGLEQUOTE, "\"", MT_STRING);
assignKey(C_SHIFT | K_COMMA, "<", MT_STRING);
assignKey(C_SHIFT | K_FULLSTOP, ">", MT_STRING);
assignKey(C_SHIFT | K_FORWARDSLASH, "?", MT_STRING);
assignKey(C_SHIFT | K_SPACE, " ", MT_STRING);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void KeyMap::loadPcKeys(void)
{
assignKey(K_F1, "SF1", MT_PREDEFINED);
assignKey(K_F2, "SF2", MT_PREDEFINED);
assignKey(K_F3, "SF3", MT_PREDEFINED);
assignKey(K_F4, "SF4", MT_PREDEFINED);
assignKey(K_F5, "SF5", MT_PREDEFINED);
assignKey(K_F6, "SF6", MT_PREDEFINED);
assignKey(K_F7, "SF7", MT_PREDEFINED);
assignKey(K_F8, "SF8", MT_PREDEFINED);
// Initialise the Ctrl map
assignKey(K_F1 | C_CTRL, "SF8", MT_PREDEFINED);
assignKey(K_F2 | C_CTRL, "SF9", MT_PREDEFINED);
assignKey(K_F3 | C_CTRL, "SF10", MT_PREDEFINED);
assignKey(K_F4 | C_CTRL, "SF11", MT_PREDEFINED);
assignKey(K_F5 | C_CTRL, "SF12", MT_PREDEFINED);
assignKey(K_F6 | C_CTRL, "SF13", MT_PREDEFINED);
assignKey(K_F7 | C_CTRL, "SF14", MT_PREDEFINED);
assignKey(K_F8 | C_CTRL, "SF15", MT_PREDEFINED);
// Initialise the Shift map
assignKey(K_F1 | C_SHIFT, "SF16", MT_PREDEFINED);
assignKey(K_F2 | C_SHIFT, "SF17", MT_PREDEFINED);
assignKey(K_F3 | C_SHIFT, "SF18", MT_PREDEFINED);
assignKey(K_F4 | C_SHIFT, "SF19", MT_PREDEFINED);
assignKey(K_F5 | C_SHIFT, "SF20", MT_PREDEFINED);
assignKey(K_F6 | C_SHIFT, "SF21", MT_PREDEFINED);
assignKey(K_F7 | C_SHIFT, "SF22", MT_PREDEFINED);
assignKey(K_F8 | C_SHIFT, "SF23", MT_PREDEFINED);
// Initialise the Alt map
assignKey(K_F1 | C_ALT, "SF24", MT_PREDEFINED);
assignKey(K_F2 | C_ALT, "SF25", MT_PREDEFINED);
assignKey(K_F3 | C_ALT, "SF26", MT_PREDEFINED);
assignKey(K_F4 | C_ALT, "SF27", MT_PREDEFINED);
assignKey(K_F5 | C_ALT, "SF28", MT_PREDEFINED);
assignKey(K_F6 | C_ALT, "SF29", MT_PREDEFINED);
assignKey(K_F7 | C_ALT, "SF30", MT_PREDEFINED);
assignKey(K_F8 | C_ALT, "SF31", MT_PREDEFINED);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void KeyMap::loadBank1Keys(void)
{
assignKey(K_F1, "SF0", MT_PREDEFINED);
assignKey(K_F2, "SF1", MT_PREDEFINED);
assignKey(K_F3, "SF2", MT_PREDEFINED);
assignKey(K_F4, "SF3", MT_PREDEFINED);
assignKey(K_F5, "SF12", MT_PREDEFINED);
assignKey(K_F6, "SF13", MT_PREDEFINED);
assignKey(K_F7, "SF14", MT_PREDEFINED);
assignKey(K_F8, "SF15", MT_PREDEFINED);
// Initialise the Ctrl map
assignKey(K_F1 | C_CTRL, "SF4", MT_PREDEFINED);
assignKey(K_F2 | C_CTRL, "SF5", MT_PREDEFINED);
assignKey(K_F3 | C_CTRL, "SF6", MT_PREDEFINED);
assignKey(K_F4 | C_CTRL, "SF7", MT_PREDEFINED);
assignKey(K_F5 | C_CTRL, "SF8", MT_PREDEFINED);
assignKey(K_F6 | C_CTRL, "SF9", MT_PREDEFINED);
assignKey(K_F7 | C_CTRL, "SF10", MT_PREDEFINED);
assignKey(K_F8 | C_CTRL, "SF11", MT_PREDEFINED);
// Initialise the Shift map
assignKey(K_F1 | C_SHIFT, "SF16", MT_PREDEFINED);
assignKey(K_F2 | C_SHIFT, "SF17", MT_PREDEFINED);
assignKey(K_F3 | C_SHIFT, "SF18", MT_PREDEFINED);
assignKey(K_F4 | C_SHIFT, "SF19", MT_PREDEFINED);
assignKey(K_F5 | C_SHIFT, "SF28", MT_PREDEFINED);
assignKey(K_F6 | C_SHIFT, "SF29", MT_PREDEFINED);
assignKey(K_F7 | C_SHIFT, "SF30", MT_PREDEFINED);
assignKey(K_F8 | C_SHIFT, "SF31", MT_PREDEFINED);
// Initialise the Alt map
assignKey(K_F1 | C_ALT, "SF20", MT_PREDEFINED);
assignKey(K_F2 | C_ALT, "SF21", MT_PREDEFINED);
assignKey(K_F3 | C_ALT, "SF22", MT_PREDEFINED);
assignKey(K_F4 | C_ALT, "SF23", MT_PREDEFINED);
assignKey(K_F5 | C_ALT, "SF24", MT_PREDEFINED);
assignKey(K_F6 | C_ALT, "SF25", MT_PREDEFINED);
assignKey(K_F7 | C_ALT, "SF26", MT_PREDEFINED);
assignKey(K_F8 | C_ALT, "SF27", MT_PREDEFINED);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void KeyMap::loadBank2Keys(void)
{
assignKey(K_F1, "SF0", MT_PREDEFINED);
assignKey(K_F2, "SF1", MT_PREDEFINED);
assignKey(K_F3, "SF2", MT_PREDEFINED);
assignKey(K_F4, "SF3", MT_PREDEFINED);
assignKey(K_F5, "SF4", MT_PREDEFINED);
assignKey(K_F6, "SF5", MT_PREDEFINED);
assignKey(K_F7, "SF6", MT_PREDEFINED);
assignKey(K_F8, "SF7", MT_PREDEFINED);
// Initialise the Ctrl map
assignKey(K_F1 | C_CTRL, "SF8", MT_PREDEFINED);
assignKey(K_F2 | C_CTRL, "SF9", MT_PREDEFINED);
assignKey(K_F3 | C_CTRL, "SF10", MT_PREDEFINED);
assignKey(K_F4 | C_CTRL, "SF11", MT_PREDEFINED);
assignKey(K_F5 | C_CTRL, "SF12", MT_PREDEFINED);
assignKey(K_F6 | C_CTRL, "SF13", MT_PREDEFINED);
assignKey(K_F7 | C_CTRL, "SF14", MT_PREDEFINED);
assignKey(K_F8 | C_CTRL, "SF15", MT_PREDEFINED);
// Initialise the Shift map
assignKey(K_F1 | C_SHIFT, "SF16", MT_PREDEFINED);
assignKey(K_F2 | C_SHIFT, "SF17", MT_PREDEFINED);
assignKey(K_F3 | C_SHIFT, "SF18", MT_PREDEFINED);
assignKey(K_F4 | C_SHIFT, "SF19", MT_PREDEFINED);
assignKey(K_F5 | C_SHIFT, "SF20", MT_PREDEFINED);
assignKey(K_F6 | C_SHIFT, "SF21", MT_PREDEFINED);
assignKey(K_F7 | C_SHIFT, "SF22", MT_PREDEFINED);
assignKey(K_F8 | C_SHIFT, "SF23", MT_PREDEFINED);
// Initialise the Alt map
assignKey(K_F1 | C_ALT, "SF24", MT_PREDEFINED);
assignKey(K_F2 | C_ALT, "SF25", MT_PREDEFINED);
assignKey(K_F3 | C_ALT, "SF26", MT_PREDEFINED);
assignKey(K_F4 | C_ALT, "SF27", MT_PREDEFINED);
assignKey(K_F5 | C_ALT, "SF28", MT_PREDEFINED);
assignKey(K_F6 | C_ALT, "SF29", MT_PREDEFINED);
assignKey(K_F7 | C_ALT, "SF30", MT_PREDEFINED);
assignKey(K_F8 | C_ALT, "SF31", MT_PREDEFINED);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void KeyMap::loadNiakwaKeys(void)
{
assignKey(K_F1, "SF0", MT_PREDEFINED);
assignKey(K_F2, "SF1", MT_PREDEFINED);
assignKey(K_F3, "SF2", MT_PREDEFINED);
assignKey(K_F4, "SF3", MT_PREDEFINED);
assignKey(K_F5, "SF4", MT_PREDEFINED);
assignKey(K_F6, "SF5", MT_PREDEFINED);
assignKey(K_F7, "SF6", MT_PREDEFINED);
assignKey(K_F8, "SF7", MT_PREDEFINED);
assignKey(K_F9, "SF8", MT_PREDEFINED);
assignKey(K_F10, "SF9", MT_PREDEFINED);
// Initialise the Shift map
assignKey(K_F1 | C_SHIFT, "SF10", MT_PREDEFINED);
assignKey(K_F2 | C_SHIFT, "SF11", MT_PREDEFINED);
assignKey(K_F3 | C_SHIFT, "SF12", MT_PREDEFINED);
assignKey(K_F4 | C_SHIFT, "SF13", MT_PREDEFINED);
assignKey(K_F5 | C_SHIFT, "SF14", MT_PREDEFINED);
assignKey(K_F6 | C_SHIFT, "SF15", MT_PREDEFINED);
// Initialise the Alt map
assignKey(K_F1 | C_ALT, "SF16", MT_PREDEFINED);
assignKey(K_F2 | C_ALT, "SF17", MT_PREDEFINED);
assignKey(K_F3 | C_ALT, "SF18", MT_PREDEFINED);
assignKey(K_F4 | C_ALT, "SF19", MT_PREDEFINED);
assignKey(K_F5 | C_ALT, "SF20", MT_PREDEFINED);
assignKey(K_F6 | C_ALT, "SF21", MT_PREDEFINED);
assignKey(K_F7 | C_ALT, "SF22", MT_PREDEFINED);
assignKey(K_F8 | C_ALT, "SF23", MT_PREDEFINED);
assignKey(K_F9 | C_ALT, "SF24", MT_PREDEFINED);
assignKey(K_F10 | C_ALT, "SF25", MT_PREDEFINED);
// Initialise the Ctrl map
assignKey(K_F1 | C_CTRL, "SF26", MT_PREDEFINED);
assignKey(K_F2 | C_CTRL, "SF27", MT_PREDEFINED);
assignKey(K_F3 | C_CTRL, "SF28", MT_PREDEFINED);
assignKey(K_F4 | C_CTRL, "SF29", MT_PREDEFINED);
assignKey(K_F5 | C_CTRL, "SF30", MT_PREDEFINED);
assignKey(K_F6 | C_CTRL, "SF31", MT_PREDEFINED);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void KeyMap::loadEmuKeys(void)
{
loadBank2Keys();
assignKey(K_HOME, "LOAD", MT_PREDEFINED);
assignKey(K_END, "RUN", MT_PREDEFINED);
assignKey(K_F2, "ACTION", MT_PREDEFINED);
assignKey(K_F10, "CANCEL", MT_PREDEFINED);
assignKey(K_INVQUOTE | C_SHIFT, "RECALL", MT_PREDEFINED);
//###Check the following two!
assignKey(K_F7, "SF7", MT_PREDEFINED);
assignKey(K_F8, "SF8", MT_PREDEFINED);
assignKey(K_S | C_CTRL, "HALT", MT_PREDEFINED);
assignKey(K_C | C_CTRL, "CONTINUE", MT_PREDEFINED);
assignKey(K_F5, "ERASE", MT_PREDEFINED);
assignKey(K_F6, "SHERASE", MT_PREDEFINED);
assignKey(K_R | C_CTRL, "RESET", MT_PREDEFINED);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// PUBLIC Methods
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
KeyMap::KeyMap()
{
// Setup the Keyboard map
_mapTable = new keyMap* [MAP_SIZE];
for (int i=0; i<MAP_SIZE; i++)
   {
   _mapTable[i] = new keyMap;
   _mapTable[i]->name[0] = '\0';
   _mapTable[i]->code = 0;
   _mapTable[i]->type = MT_STRING;
   _mapTable[i]->preMapIndex = -1;
   _mapTable[i]->codeLength = 0;
   }

_fileName = new string();
// Load a default keytable
loadDefault();
setKeyNames();
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
KeyMap::~KeyMap()
{
for (int i=0; i<MAP_SIZE; i++)
   {
   delete _mapTable[i]->code;
   delete _mapTable[i];
   }
delete [] _mapTable;
delete _fileName;
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
// Pre: code should be NULL terminated
//---------------------------------------------------------------------------
void KeyMap::assignKey(const unsigned int key,
   unsigned char *code,
   type_t type,
   int preMapIndex)
{

int length;

if (type == MT_PREDEFINED)
   {
   length = MAX_PCODE;
   unsigned char preMapCode[MAX_PCODE] = "";

   if (preMapIndex == -1)
      preMapIndex = getPreMapIndex(code);
   getPreMapCode(preMapIndex, preMapCode, length);
   delete _mapTable[key]->code;
   _mapTable[key]->code = new unsigned char[length];
   memcpy(_mapTable[key]->code, preMapCode, length);
   _mapTable[key]->type = type;
   _mapTable[key]->preMapIndex = preMapIndex;
   _mapTable[key]->codeLength = length;
   }
else if (type == MT_STRING)
   {
   length = min((unsigned int)MAX_CODE, strlen((char *)code));

   delete _mapTable[key]->code;
   _mapTable[key]->code = new unsigned char[length];
   memcpy(_mapTable[key]->code, code, length);
   _mapTable[key]->codeLength = length;
   _mapTable[key]->type = type;
   _mapTable[key]->preMapIndex = -1;
   }
else if (type == MT_HEX)
   {
   length = min((unsigned int)MAX_CODE*2, strlen((char *)code));
   unsigned char hexBuf[MAX_CODE*2] = "";

   strToHexBuf(code, hexBuf, length);
   delete _mapTable[key]->code;
   _mapTable[key]->code = new unsigned char[length];
   memcpy(_mapTable[key]->code, hexBuf, length);
   _mapTable[key]->codeLength = length;
   _mapTable[key]->type = type;
   _mapTable[key]->preMapIndex = -1;
   }
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes: OK
//---------------------------------------------------------------------------
int KeyMap::getPreMapIndex(const char *name)
{
// Return 0 on failure (to access the "NONE" item in the premap),
// otherwise the index of the string
int position=0;

for (int i=1; i<PREMAP_SIZE; i++)
   if (strncmp(name, preMap[i].name, MAX_PNAME) == 0)
      {
      position = i;
      break;
      }
return (position);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes: OK
// Pre: name cannot be null,
//    name must point to a string large enough to hold MAX_PNAME+1 characters
//---------------------------------------------------------------------------
void KeyMap::getPreMapName(const int index, char *name)
{
if (index >= 0 && index <= PREMAP_SIZE)
   {
   strncpy(name, preMap[index].name, MAX_PNAME);
   name[MAX_PNAME] = '\0';
   }
else
   name[0] = '\0';
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes: OK
// Pre: code must allow for MAX_PCODE characters
//---------------------------------------------------------------------------
void KeyMap::getPreMapCode(const int index, unsigned char *code, int &length)
{
if (length > 0 && index >= 0 && index <= PREMAP_SIZE)
   {
   length = min(length, preMap[index].codeLength);
   memcpy(code, preMap[index].code, length);
   }
else
   {
   length = 0;
   code = '\0';
   }
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes: OK
// Pre: code must allow for MAX_PCODE characters
//---------------------------------------------------------------------------
void KeyMap::getPreMapCode(const char *name, unsigned char *code, int &length)
{
int index = getPreMapIndex(name);

getPreMapCode(index, code, length);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes: OK
// Pre: name must allow for MAX_NAME+1 characters
//---------------------------------------------------------------------------
void KeyMap::getKeyName(const unsigned int key, const shiftState_t sState, char *name)
{
if (sState.alt)
   strncpy(name, _mapTable[key | C_ALT]->name, MAX_NAME);
else if (sState.ctrl)
   strncpy(name, _mapTable[key | C_CTRL]->name, MAX_NAME);
else if (sState.shift)
   strncpy(name, _mapTable[key | C_SHIFT]->name, MAX_NAME);
else
   strncpy(name, _mapTable[key]->name, MAX_NAME);
name[MAX_NAME] = '\0';
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes: OK
//---------------------------------------------------------------------------
type_t KeyMap::getKeyType(const unsigned int key, const shiftState_t sState)
{
if (sState.alt)
   return (_mapTable[key | C_ALT]->type);
else if (sState.ctrl)
   return (_mapTable[key | C_CTRL]->type);
else if (sState.shift)
   return (_mapTable[key | C_SHIFT]->type);
else
   return (_mapTable[key]->type);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes: OK
//---------------------------------------------------------------------------
int KeyMap::getKeyPreMapIndex(const unsigned int key, const shiftState_t sState)
{
if (sState.alt)
   return (_mapTable[key | C_ALT]->preMapIndex);
else if (sState.ctrl)
   return (_mapTable[key | C_CTRL]->preMapIndex);
else if (sState.shift)
   return (_mapTable[key | C_SHIFT]->preMapIndex);
else
   return (_mapTable[key]->preMapIndex);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes: OK
//---------------------------------------------------------------------------
void KeyMap::setKeyNames(void)
{
string prefix, name;
controlConstant offset;

for (int i=0; i<=3; i++)
   {
   if (i == 0)
      {
      prefix = "";
      offset = C_UNSHIFT;
      }
   else if (i == 1)
      {
      prefix = "SHIFT+";
      offset = C_SHIFT;
      }
   else if (i == 2)
      {
      prefix = "CTRL+";
      offset = C_CTRL;
      }
   else if (i == 3)
      {
      prefix = "ALT+";
      offset = C_ALT;
      }
   // Assign each key code a name
   name = prefix + "BACK_SPACE";
   strncpy(_mapTable[offset | K_BACK]->name, name.c_str(), MAX_NAME);
   name = prefix + "TAB";
   strncpy(_mapTable[offset | K_TAB]->name, name.c_str(), MAX_NAME);
   name = prefix + "CLEAR";   //###Unsure if this key is needed
   strncpy(_mapTable[offset | K_CLEAR]->name, name.c_str(), MAX_NAME);
   name = prefix + "RETURN";
   strncpy(_mapTable[offset | K_RETURN]->name, name.c_str(), MAX_NAME);
   name = prefix + "ESCAPE";
   strncpy(_mapTable[offset | K_ESCAPE]->name, name.c_str(), MAX_NAME);
   name = prefix + "SPACE";
   strncpy(_mapTable[offset | K_SPACE]->name, name.c_str(), MAX_NAME);
   name = prefix + "PRIOR";
   strncpy(_mapTable[offset | K_PRIOR]->name, name.c_str(), MAX_NAME);
   name = prefix + "NEXT";
   strncpy(_mapTable[offset | K_NEXT]->name, name.c_str(), MAX_NAME);
   name = prefix + "END";
   strncpy(_mapTable[offset | K_END]->name, name.c_str(), MAX_NAME);
   name = prefix + "HOME";
   strncpy(_mapTable[offset | K_HOME]->name, name.c_str(), MAX_NAME);
   name = prefix + "LEFT";
   strncpy(_mapTable[offset | K_LEFT]->name, name.c_str(), MAX_NAME);
   name = prefix + "UP";
   strncpy(_mapTable[offset | K_UP]->name, name.c_str(), MAX_NAME);
   name = prefix + "RIGHT";
   strncpy(_mapTable[offset | K_RIGHT]->name, name.c_str(), MAX_NAME);
   name = prefix + "DOWN";
   strncpy(_mapTable[offset | K_DOWN]->name, name.c_str(), MAX_NAME);
   name = prefix + "SELECT";  //###Unsure if this key is needed
   strncpy(_mapTable[offset | K_SELECT]->name, name.c_str(), MAX_NAME);
   name = prefix + "EXECUTE"; //###Unsure if this key is needed
   strncpy(_mapTable[offset | K_EXECUTE]->name, name.c_str(), MAX_NAME);
   name = prefix + "INSERT";
   strncpy(_mapTable[offset | K_INSERT]->name, name.c_str(), MAX_NAME);
   name = prefix + "DELETE";
   strncpy(_mapTable[offset | K_DELETE]->name, name.c_str(), MAX_NAME);
   name = prefix + "0";
   strncpy(_mapTable[offset | K_0]->name, name.c_str(), MAX_NAME);
   name = prefix + "1";
   strncpy(_mapTable[offset | K_1]->name, name.c_str(), MAX_NAME);
   name = prefix + "2";
   strncpy(_mapTable[offset | K_2]->name, name.c_str(), MAX_NAME);
   name = prefix + "3";
   strncpy(_mapTable[offset | K_3]->name, name.c_str(), MAX_NAME);
   name = prefix + "4";
   strncpy(_mapTable[offset | K_4]->name, name.c_str(), MAX_NAME);
   name = prefix + "5";
   strncpy(_mapTable[offset | K_5]->name, name.c_str(), MAX_NAME);
   name = prefix + "6";
   strncpy(_mapTable[offset | K_6]->name, name.c_str(), MAX_NAME);
   name = prefix + "7";
   strncpy(_mapTable[offset | K_7]->name, name.c_str(), MAX_NAME);
   name = prefix + "8";
   strncpy(_mapTable[offset | K_8]->name, name.c_str(), MAX_NAME);
   name = prefix + "9";
   strncpy(_mapTable[offset | K_9]->name, name.c_str(), MAX_NAME);
   name = prefix + "A";
   strncpy(_mapTable[offset | K_A]->name, name.c_str(), MAX_NAME);
   name = prefix + "B";
   strncpy(_mapTable[offset | K_B]->name, name.c_str(), MAX_NAME);
   name = prefix + "C";
   strncpy(_mapTable[offset | K_C]->name, name.c_str(), MAX_NAME);
   name = prefix + "D";
   strncpy(_mapTable[offset | K_D]->name, name.c_str(), MAX_NAME);
   name = prefix + "E";
   strncpy(_mapTable[offset | K_E]->name, name.c_str(), MAX_NAME);
   name = prefix + "F";
   strncpy(_mapTable[offset | K_F]->name, name.c_str(), MAX_NAME);
   name = prefix + "G";
   strncpy(_mapTable[offset | K_G]->name, name.c_str(), MAX_NAME);
   name = prefix + "H";
   strncpy(_mapTable[offset | K_H]->name, name.c_str(), MAX_NAME);
   name = prefix + "I";
   strncpy(_mapTable[offset | K_I]->name, name.c_str(), MAX_NAME);
   name = prefix + "J";
   strncpy(_mapTable[offset | K_J]->name, name.c_str(), MAX_NAME);
   name = prefix + "K";
   strncpy(_mapTable[offset | K_K]->name, name.c_str(), MAX_NAME);
   name = prefix + "L";
   strncpy(_mapTable[offset | K_L]->name, name.c_str(), MAX_NAME);
   name = prefix + "M";
   strncpy(_mapTable[offset | K_M]->name, name.c_str(), MAX_NAME);
   name = prefix + "N";
   strncpy(_mapTable[offset | K_N]->name, name.c_str(), MAX_NAME);
   name = prefix + "O";
   strncpy(_mapTable[offset | K_O]->name, name.c_str(), MAX_NAME);
   name = prefix + "P";
   strncpy(_mapTable[offset | K_P]->name, name.c_str(), MAX_NAME);
   name = prefix + "Q";
   strncpy(_mapTable[offset | K_Q]->name, name.c_str(), MAX_NAME);
   name = prefix + "R";
   strncpy(_mapTable[offset | K_R]->name, name.c_str(), MAX_NAME);
   name = prefix + "S";
   strncpy(_mapTable[offset | K_S]->name, name.c_str(), MAX_NAME);
   name = prefix + "T";
   strncpy(_mapTable[offset | K_T]->name, name.c_str(), MAX_NAME);
   name = prefix + "U";
   strncpy(_mapTable[offset | K_U]->name, name.c_str(), MAX_NAME);
   name = prefix + "V";
   strncpy(_mapTable[offset | K_V]->name, name.c_str(), MAX_NAME);
   name = prefix + "W";
   strncpy(_mapTable[offset | K_W]->name, name.c_str(), MAX_NAME);
   name = prefix + "X";
   strncpy(_mapTable[offset | K_X]->name, name.c_str(), MAX_NAME);
   name = prefix + "Y";
   strncpy(_mapTable[offset | K_Y]->name, name.c_str(), MAX_NAME);
   name = prefix + "Z";
   strncpy(_mapTable[offset | K_Z]->name, name.c_str(), MAX_NAME);
   name = prefix + "NUMPAD0";
   strncpy(_mapTable[offset | K_NUMPAD0]->name, name.c_str(), MAX_NAME);
   name = prefix + "NUMPAD1";
   strncpy(_mapTable[offset | K_NUMPAD1]->name, name.c_str(), MAX_NAME);
   name = prefix + "NUMPAD2";
   strncpy(_mapTable[offset | K_NUMPAD2]->name, name.c_str(), MAX_NAME);
   name = prefix + "NUMPAD3";
   strncpy(_mapTable[offset | K_NUMPAD3]->name, name.c_str(), MAX_NAME);
   name = prefix + "NUMPAD4";
   strncpy(_mapTable[offset | K_NUMPAD4]->name, name.c_str(), MAX_NAME);
   name = prefix + "NUMPAD5";
   strncpy(_mapTable[offset | K_NUMPAD5]->name, name.c_str(), MAX_NAME);
   name = prefix + "NUMPAD6";
   strncpy(_mapTable[offset | K_NUMPAD6]->name, name.c_str(), MAX_NAME);
   name = prefix + "NUMPAD7";
   strncpy(_mapTable[offset | K_NUMPAD7]->name, name.c_str(), MAX_NAME);
   name = prefix + "NUMPAD8";
   strncpy(_mapTable[offset | K_NUMPAD8]->name, name.c_str(), MAX_NAME);
   name = prefix + "NUMPAD9";
   strncpy(_mapTable[offset | K_NUMPAD9]->name, name.c_str(), MAX_NAME);
   name = prefix + "MULTIPLY";
   strncpy(_mapTable[offset | K_MULTIPLY]->name, name.c_str(), MAX_NAME);
   name = prefix + "ADD";
   strncpy(_mapTable[offset | K_ADD]->name, name.c_str(), MAX_NAME);
   name = prefix + "SEPARATOR";  //###Unsure if this key is needed
   strncpy(_mapTable[offset | K_SEPARATOR]->name, name.c_str(), MAX_NAME);
   name = prefix + "SUBTRACT";
   strncpy(_mapTable[offset | K_SUBTRACT]->name, name.c_str(), MAX_NAME);
   name = prefix + "DECIMAL";
   strncpy(_mapTable[offset | K_DECIMAL]->name, name.c_str(), MAX_NAME);
   name = prefix + "DIVIDE";
   strncpy(_mapTable[offset | K_DIVIDE]->name, name.c_str(), MAX_NAME);
   name = prefix + "F1";
   strncpy(_mapTable[offset | K_F1]->name, name.c_str(), MAX_NAME);
   name = prefix + "F2";
   strncpy(_mapTable[offset | K_F2]->name, name.c_str(), MAX_NAME);
   name = prefix + "F3";
   strncpy(_mapTable[offset | K_F3]->name, name.c_str(), MAX_NAME);
   name = prefix + "F4";
   strncpy(_mapTable[offset | K_F4]->name, name.c_str(), MAX_NAME);
   name = prefix + "F5";
   strncpy(_mapTable[offset | K_F5]->name, name.c_str(), MAX_NAME);
   name = prefix + "F6";
   strncpy(_mapTable[offset | K_F6]->name, name.c_str(), MAX_NAME);
   name = prefix + "F7";
   strncpy(_mapTable[offset | K_F7]->name, name.c_str(), MAX_NAME);
   name = prefix + "F8";
   strncpy(_mapTable[offset | K_F8]->name, name.c_str(), MAX_NAME);
   name = prefix + "F9";
   strncpy(_mapTable[offset | K_F9]->name, name.c_str(), MAX_NAME);
   name = prefix + "F10";
   strncpy(_mapTable[offset | K_F10]->name, name.c_str(), MAX_NAME);
   name = prefix + "F11";
   strncpy(_mapTable[offset | K_F11]->name, name.c_str(), MAX_NAME);
   name = prefix + "F12";
   strncpy(_mapTable[offset | K_F12]->name, name.c_str(), MAX_NAME);
   //###Note, the following codes MAY vary on other machines (different keyboards)
   name = prefix + "SEMICOLON";
   strncpy(_mapTable[offset | K_SEMICOLON]->name, name.c_str(), MAX_NAME);
   name = prefix + "EQUALS";
   strncpy(_mapTable[offset | K_EQUALS]->name, name.c_str(), MAX_NAME);
   name = prefix + "COMMA";
   strncpy(_mapTable[offset | K_COMMA]->name, name.c_str(), MAX_NAME);
   name = prefix + "MINUS";
   strncpy(_mapTable[offset | K_MINUS]->name, name.c_str(), MAX_NAME);
   name = prefix + "FULLSTOP";
   strncpy(_mapTable[offset | K_FULLSTOP]->name, name.c_str(), MAX_NAME);
   name = prefix + "FORWARDSLASH";
   strncpy(_mapTable[offset | K_FORWARDSLASH]->name, name.c_str(), MAX_NAME);
   name = prefix + "INVQUOTE";
   strncpy(_mapTable[offset | K_INVQUOTE]->name, name.c_str(), MAX_NAME);
   name = prefix + "SQUARELEFT";
   strncpy(_mapTable[offset | K_SQUARELEFT]->name, name.c_str(), MAX_NAME);
   name = prefix + "BACKSLASH";
   strncpy(_mapTable[offset | K_BACKSLASH]->name, name.c_str(), MAX_NAME);
   name = prefix + "SQUARERIGHT";
   strncpy(_mapTable[offset | K_SQUARERIGHT]->name, name.c_str(), MAX_NAME);
   name = prefix + "SINGLEQUOTE";
   strncpy(_mapTable[offset | K_SINGLEQUOTE]->name, name.c_str(), MAX_NAME);
   }
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes: OK
//---------------------------------------------------------------------------
void KeyMap::loadDefault(default_t defaultType)
{
// Initialize the Key codes (leave the Key Names)
for (int i=0; i<MAP_SIZE; i++)
   {
   delete _mapTable[i]->code;
   _mapTable[i]->code = 0;
   _mapTable[i]->type = MT_STRING;
   _mapTable[i]->preMapIndex = -1;
   _mapTable[i]->codeLength = 0;
   }
loadBasicKeys();
switch (defaultType)
   {
   case DT_BANK1 :
      *_fileName = "<BANK1>";
      loadBank1Keys();
   break;
   case DT_BANK2 :
      *_fileName = "<BANK2>";
      loadBank2Keys();
   break;
   case DT_PC :
      *_fileName = "<PC-EQUIV>";
      loadPcKeys();
   break;
   case DT_NIAKWA :
      *_fileName = "<NIAKWA>";
      loadNiakwaKeys();
   break;
   default : // DT_EMU
      *_fileName = "<EMU Default>";
      loadEmuKeys();
   break;
   }
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes: Assumes MAX_CODE > MAX_NAME
//---------------------------------------------------------------------------
bool KeyMap::loadSetup(const string &fileName)
{
ifstream inFile;
char tempCode[MAX_CODE*2+1] = "";
char codeType;
int key;
int inCnt = 0;

inFile.open(fileName.c_str(), ios::nocreate | ios::binary);
if (!inFile)
   {
   return (false);
   }
// Read the header information and ensure we have a valid keymap file
inFile.get(tempCode, HEADER_LENGTH);
if (strncmp(tempCode, KEYMAP_HEADER, HEADER_LENGTH) != 0)
   return (false);
loadDefault();

// Ignore the endl character and the text name.
while (inFile.ignore(MAX_NAME+2, '=') && !inFile.eof())
   {
   // Clear the string
   tempCode[0] = '\0';
   inFile >> key;             // key table index
   inFile.ignore(1 , ',');    // ignore the comma
   inFile >> codeType;        // type of key that is defined
   switch (codeType)
      {
      case HEX_DELIMETER:
         inFile.get(tempCode, MAX_CODE*2);
         assignKey(key, tempCode, MT_HEX);
      break;
      case STR_DELIMETER:
         inFile.get(tempCode, MAX_CODE);
         inCnt = inFile.gcount();
         tempCode[inCnt-1] = '\0';           // Remove the ending double quote
         assignKey(key, tempCode, MT_STRING);
      break;
      case PRE_DELIMETER:
         inFile.get(tempCode, MAX_PNAME+1);  // get always appends a null
         assignKey(key, tempCode, MT_PREDEFINED);
      break;
      }
   }
inFile.close();
*_fileName = fileName;
return (true);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
bool KeyMap::saveSetup(const string &fileName)
{
ofstream outFile;
char name[MAX_PNAME+1] = "";

outFile.open(fileName.c_str(), ios::trunc | ios::binary);
if (!outFile)
   {
   return (false);
   }
   // Output the header
   outFile << KEYMAP_HEADER << endl;
   // Output the rest of the keymaps
   for (int i=0; i<MAP_SIZE; i++)
      {
      if (_mapTable[i]->codeLength == 0)
         continue;
      // The code is not empty
      outFile << _mapTable[i]->name << "=" << i << ",";
      switch (_mapTable[i]->type)
         {
         case MT_PREDEFINED:
            outFile << PRE_DELIMETER;
            getPreMapName(_mapTable[i]->preMapIndex, name);
            outFile << name;
         break;
         case MT_STRING:
            outFile << STR_DELIMETER;
            outFile.write(_mapTable[i]->code, _mapTable[i]->codeLength);
            outFile << STR_DELIMETER;
         break;
         case MT_HEX:
            int length = _mapTable[i]->codeLength;
            char hexString[MAX_CODE*2];

            hexBufToStr(_mapTable[i]->code, hexString, length);
            outFile << HEX_DELIMETER;
            outFile.write(hexString, length);
         break;
         }
      outFile << endl;
      }
outFile.close();
*_fileName = fileName;
return (true);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes: Order of precedence is ALT, CTRL, SHIFT
//---------------------------------------------------------------------------
void KeyMap::getKeyCode(const unsigned int key, const shiftState_t sState,
   unsigned char *code, int &length)
{
// Check for the control keys.
controlConstant offset;

if (sState.alt)
   offset = C_ALT;
else if (sState.ctrl)
   offset = C_CTRL;
else if (sState.shift)
   offset = C_SHIFT;
else
   offset = C_UNSHIFT;

length = min(length, _mapTable[key | offset]->codeLength);
if (length > 0)
   memcpy(code, _mapTable[key | offset]->code, length);
else
   code[0] = '\0';
}

void KeyMap::getFileName(string &fileName)
{
fileName = *_fileName;
}
//---------------------------------------------------------------------------
// END OF FILE
//---------------------------------------------------------------------------


