//---------------------------------------------------------------------------
// Copyright Mark Pickersgill 2001-2014
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
#ifndef InterpreterH
#define InterpreterH
//---------------------------------------------------------------------------
// File: Interpreter.h
// Desc.: Header file for the based class implementation of the 
//    Interpreter class.
//
//---------------------------------------------------------------------------
//  Created: 17/08/2001
//  Version History:
//    17/08/2001, 0.0.0 - Basic class outline
//    24/01/2002, 0.0.3 - Added new return type (iFinishContinue) to allow
//       further processing of the next immediate character in the buffer.
//    14/02/2002, 0.0.4 - Moved static variables from function checking
//       routines into the main class so that the state of the object can be
//       reset at any time.
//       - Implemented the reset function
//---------------------------------------------------------------------------
// TODO:
//---------------------------------------------------------------------------
// Required CUSTOM header files
//---------------------------------------------------------------------------
#include "FileInterface.h"
#include "ScreenInterface.h"
#include "CommInterface.h"
#include "PrinterInterface.h"
//---------------------------------------------------------------------------
// Required SYSTEM header files
//---------------------------------------------------------------------------
#include <fstream.h>
#include <string>

//---------------------------------------------------------------------------
// --== DECLARATIONS ==--
//---------------------------------------------------------------------------
//
// Specifies the return type for any control code handlers
enum iResult { iFinish, iNone, iError, iContinue, iFinishContinue };
// iFinish - A found sequence was successfully processed
// iNone - No function code was found matching the input code
// iError - A function code was started, however the parameters were invalid
// iContinue - A function code was started and there is still more parameters to
//    process.
// iFinishContinue - Specific to the Hex(FB) functions. If we have finished
// processing the HEX(FB) sequence but the next character should NOT be checked for
// a new HEX(FB) sequence then this is set

// Enums that are being compared to unsigned characters need to be fully initialized with
// 0xHHHH so that all of the bytes in the integer variable are used.
enum control1_t { FB=0x00FB, CONTROL1_CHECK };

enum control2_t { DATA_BYTE=0x00D0, COMPRESS=0x0060, RSD_START=0x00D3,
   RSD_ABORT=0x00D4, RSD_GO=0x00D8, RSD_STOP=0x00DA, SEL_SCREEN=0x00F0,
   SEL_PRINTER=0x00F1, RESET_TERM=0x00F2, AUX_PARAM=0x00F3,
   CURSOR_BLINK2=0x00F4, AUX_SWITCH=0x00F5, CLEAR_CRT=0x00F6, AUX_FORMAT=0x00F7,
   CURSOR_STEADY=0x00F8, CURSOR_BLINK3=0x00FC, DELAY=0x00C0, NEW_FB=0x00FB,
   CONTROL2_OUTPUT=0x00FE, CONTROL2_CHECK=0x00FF };

enum screen1_t { NONE=0x0000, HOME=0x0001, MULTI=0x0002, CLEAR=0x0003,
   RESERVED1=0x0004, CURSOR_ON=0x0005, CURSOR_OFF=0x0006, ALARM=0x0007,
   LEFT=0x0008, RIGHT=0x0009, DOWN=0x000A, RESERVED2=0x000B, UP=0x000C,
   START_LINE=0x000D, ATTRIB_ON=0x000E, ATTRIB_OFF=0x000F, SCREEN1_OUTPUT,
   SCREEN1_CHECK };

enum screen2_t { ZERO=0x0000, OPEN_WIN=0x0001, CHAR_SET=0x0002, COLOURS=0x0003,
   ATTRIBS=0x0004, CURSOR_BLINK=0x0005, SCREEN_MODE=0x0006, SELF_ID=0x0008,
   SCREEN_WRITE=0x0009, DRAW_BOX=0x000B, DRAW_GRAPHIC=0x000C, INIT=0x000D,
   KEYBOARD=0x000E, FB_EQUIV=0x001B, SCREEN2_OUTPUT, SCREEN2_CHECK };

enum screen3_t { WT_COLOUR=0x0002, WT_STATUS_CLEAR=0x0005,
   WT_STATUS_WRITE=0x000A, FT_OTHER=0x0006, SCREEN3_OUTPUT, SCREEN3_CHECK };

enum printer1_t { CONTROL=0x0002, PRINTER1_OUTPUT, PRINTER1_CHECK };

enum printer2_t { PRINTER_ZERO=0x0000, PRINTER2_OUTPUT, PRINTER2_CHECK };

enum printer3_t { CREATE_FILE=0x000C, TEST_FILE=0x000B, REROUTE=0x0008,
   RESTORE=0x0004, PRINTER3_OUTPUT, PRINTER3_CHECK };

enum paramLevel_t { PARAM1, PARAM2, PARAM3, PARAM4 };

enum device_t { DEV_SCREEN, DEV_PRINTER };

// Flow control for a WANG/Fasstcom terminal
#define FLOW_OFF_STR "\xFA\xFB"
#define FLOW_ON_STR "\xF8\xF9"

//---------------------------------------------------------------------------
// Class: Interpreter
// Desc.: Generic Interpreter Class
//---------------------------------------------------------------------------
class Interpreter
{
private :
   bool _flowOff;
public :
   Interpreter() { _flowOff = false; }
   virtual bool interpret(const unsigned char *buffer, int count)=0;
   virtual void setFlowOn() { _flowOff = false; }
   virtual void setFlowOff() { _flowOff = true; }
   bool isFlowOff() { return _flowOff; }
};

//---------------------------------------------------------------------------
// Class: WtInterpreter
// Desc.: WT specific Interpreter Class
//---------------------------------------------------------------------------
class WtInterpreter : public Interpreter
{
private :
   ScreenInterface *_screenInterface;
   FileInterface *_fileInterface;
   PrinterInterface *_printerInterface;
   CommInterface *_commInterface;
   bool _canInterpret;
   bool _toggleOn;
   bool _boxControlFlag;
   unsigned char *_buffer;
   int _position;
   int _count;
   device_t _outDevice;
   // Parameters used by the Screen Manipulation Functions
   // Function Parameter Levels: Which parameter a function is expecting to process
   paramLevel_t _screenParamLevel;
   paramLevel_t _controlParamLevel;
   paramLevel_t _printerParamLevel;
   // Function Levels: What function is to be called and at what level we are
   // at in the checking process
   screen1_t _screenFuncLevel1;
   screen3_t _screenFuncLevel3;
   screen2_t _screenFuncLevel2;
   control1_t _controlFuncLevel1;
   control2_t _controlFuncLevel2;

   // The following variables store the state at which either the screen or
   // printer are in with regard to the control sequence. This is not needed for
   // the hex(02) sequence of codes, as they are ONLY used for the screen.
   // Screen
   control1_t _scrControlFuncLevel1;
   control2_t _scrControlFuncLevel2;
   paramLevel_t _scrControlParamLevel;
   unsigned char _scrControlParam1;
   unsigned char _scrControlParam2;
   unsigned char _scrControlParam3;
   unsigned char _scrControlParam4;
   // Printer
   control1_t _prnControlFuncLevel1;
   control2_t _prnControlFuncLevel2;
   paramLevel_t _prnControlParamLevel;
   unsigned char _prnControlParam1;
   unsigned char _prnControlParam2;
   unsigned char _prnControlParam3;
   unsigned char _prnControlParam4;

   printer1_t _printerFuncLevel1;
   printer2_t _printerFuncLevel2;
   printer3_t _printerFuncLevel3;
   // Some miscellaneus parameter storage for the functions
   unsigned char _screenParam1;
   unsigned char _screenParam2;
   unsigned char _screenParam3;
   unsigned char _screenParam4;
   unsigned char _controlParam1;
   unsigned char _controlParam2;
   unsigned char _controlParam3;
   unsigned char _controlParam4;
protected :
   void initVariables();
   void checkStatus();
   // Level 1 Screen Manipulation Functions
   iResult screenOutputChar();
   iResult screenHome();
   iResult screenClear();
   iResult screenCursorOn();
   iResult screenCursorOff();
   iResult screenAlarm();
   iResult screenLeft();
   iResult screenRight();
   iResult screenDown();
   iResult screenUp();
   iResult screenStartLine();
   iResult screenAttribOn();
   iResult screenAttribOff();
   // Level 2 Screen Manipulation Functions
   iResult screenZero();
   iResult screenOpenWin();
   iResult screenCharSet();
   iResult screenColours();
   iResult screenAttribs();
   iResult screenCursorBlink();
   iResult screenScreenMode();
   iResult screenSelfId();
   iResult screenScreenWrite();
   iResult screenDrawBox();
   iResult screenDrawGraphic();
   iResult screenInit();
   iResult screenKeyboard();
   iResult screenFbEquiv();
   // Level 3 Screen Manipulation Functions
   iResult screenFtOther();
   iResult screenWtColour();
   iResult screenWtStatusWrite();
   iResult screenWtStatusClear();
   // Level 2 Control Functions
   iResult controlDataByte();
   iResult controlCompress();
   iResult controlRsdStart();
   iResult controlRsdAbort();
   iResult controlRsdGo();
   iResult controlRsdStop();
   iResult controlSelScreen();
   iResult controlSelPrinter();
   iResult controlResetTerm();
   iResult controlAuxParam();
   iResult controlCursorBlink2();
   iResult controlAuxSwitch();
   iResult controlClearCrt();
   iResult controlAuxFormat();
   iResult controlCursorSteady();
   iResult controlCursorBlink3();
   iResult controlDelay();
   iResult controlNewFb();
   // Level 1 Printer Functions
   iResult printerOutputChar();
   // Level 3 Printer Functions
   iResult printerWtCreateFile();
   iResult printerWtTestFile();
   iResult printerWtReroute();
   iResult printerWtRestore();
public :
   WtInterpreter();
   virtual bool interpret(const unsigned char *buffer, int count);
   virtual void setFlowOn();
   virtual void setFlowOff();
   virtual void setScreenInterface(ScreenInterface *screenInterface);
   virtual void setFileInterface(FileInterface *fileInterface);
   virtual void setPrinterInterface(PrinterInterface *printerInterface);
   virtual void setCommInterface(CommInterface *commInterface);
   // Level 1 handling functions
   virtual iResult handleControlCodes1();
   virtual iResult handleScreenCodes1();
   virtual iResult handlePrinterCodes1();
   // Level 2 handling functions
   virtual iResult handleScreenCodes2();
   virtual iResult handleControlCodes2();
   virtual iResult handlePrinterCodes2();
   // Level 3 handling functions
   virtual iResult handlePrinterCodes3();
   // Level 1 checking functions
   virtual control1_t checkControlCode1(unsigned char code);
   virtual screen1_t checkScreenCode1(unsigned char code);
   virtual printer1_t checkPrinterCode1(unsigned char code);
   // Level 2 checking functions
   virtual control2_t checkControlCode2(unsigned char code);
   virtual screen2_t checkScreenCode2(unsigned char code);
   virtual printer2_t checkPrinterCode2(unsigned char code);
   // Level 3 checking functions
   virtual printer3_t checkPrinterCode3(unsigned char code);
};
#endif
//---------------------------------------------------------------------------
// END OF FILE
//---------------------------------------------------------------------------
