//---------------------------------------------------------------------------
// Copyright Mark Pickersgill 2001-2014
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// File: Interpreter.cpp
// Desc.: Implementation file for the Generic Interpreter class.
//
//---------------------------------------------------------------------------
// Created: 17/08/2001
// Version History:
//    17/08/2001, 0.0.0 - Created the file
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// Required CUSTOM header files
//---------------------------------------------------------------------------
#include "Interpreter.h"
//---------------------------------------------------------------------------
// Required SYSTEM header files
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// --== IMPLEMENTATION ==--
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// Class: WtInterpreter
// Desc.: WT Specific Interpreter Class
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// Method: checkStatus()
// Desc.: Checks to see if translation of a buffer can be done
// Return: None
// Notes:
//---------------------------------------------------------------------------
void WtInterpreter::checkStatus()
{
if ((_fileInterface != 0) && (_screenInterface != 0) &&
   (_commInterface != 0) && (_printerInterface != 0))
   {
   // All modules are linked, so interpretation of a buffer can be done
   _canInterpret = true;
   }
else
   {
   _canInterpret = false;
   }
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::screenOutputChar()
{
// This routine should either succeed or fail
_screenInterface->putChar(_buffer[_position]);
return (iFinish);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::screenHome()
{
_screenInterface->moveCaret(DIR_TOP_HOME);
return (iFinish);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::screenClear()
{
_screenInterface->clearScreen();
_screenInterface->moveCaret(DIR_TOP_HOME);
return (iFinish);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::screenCursorOn()
{
_screenInterface->setCaretOn(true);
return (iFinish);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::screenCursorOff()
{
_screenInterface->setCaretOn(false);
return (iFinish);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::screenAlarm()
{
_screenInterface->beep();
return (iFinish);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::screenLeft()
{
_screenInterface->moveCaret(DIR_LEFT);
return (iFinish);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::screenRight()
{
_screenInterface->moveCaret(DIR_RIGHT);
return (iFinish);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::screenDown()
{
_screenInterface->moveCaret(DIR_DOWN);
return (iFinish);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::screenUp()
{
_screenInterface->moveCaret(DIR_UP);
return (iFinish);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::screenStartLine()
{
// If the attributes were toggled previously then remove them
if (_toggleOn)
   {
   _screenInterface->toggleAttribs(false);
   _toggleOn = false;
   }
_screenInterface->moveCaret(DIR_HOME);
return (iFinish);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::screenAttribOn()
{
_screenInterface->toggleAttribs(true);
_toggleOn = true;
return (iFinish);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::screenAttribOff()
{
_screenInterface->toggleAttribs(false);
_toggleOn = false;
return (iFinish);
}

//---------------------------------------------------------------------------
//
//    Level 2 Screen Manipulation Functions
//
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::screenZero()
{
//static screen3_t _screenFuncLevel3 = SCREEN3_CHECK;
iResult result;

switch (_screenFuncLevel3)
   {
   case SCREEN3_CHECK : // See which sub-function it is
      switch (_buffer[_position])
         {
         case FT_OTHER :         // FT, either save/restore screen or print on
                                 // 25th line
         case WT_COLOUR :        // WT style Colour control
         case WT_STATUS_WRITE :  // WT style status line display
         case WT_STATUS_CLEAR :  // WT style status line clear
            _screenFuncLevel3 = (screen3_t)_buffer[_position];
            result = iContinue;
         break;
         default :   // Unknown code
            result = iNone;
         break;
         }
   break;
   case FT_OTHER :         // FT, either save/restore screen or print on
                           // 25th line
      result = screenFtOther();
   break;
   case WT_COLOUR :        // WT style Colour control
      result = screenWtColour();
   break;
   case WT_STATUS_WRITE :  // WT style status line display
      result = screenWtStatusWrite();
   break;
   case WT_STATUS_CLEAR :  // WT style status line clear
      result = screenWtStatusClear();
   break;
   default :   // Error!
      result = iError;
   break;
   }
// Ensure the state is reset
if (result != iContinue)
   {
   _screenFuncLevel3 = SCREEN3_CHECK;
   }
return (result);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::screenOpenWin()
{
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::screenCharSet()
{
iResult result;

switch (_screenParamLevel)
   {
   case PARAM1 :  // Character set type
      _screenParam1 = _buffer[_position];
      _screenParamLevel = PARAM2;
      result = iContinue;
   break;
   case PARAM2 :
      _screenParam2 = _buffer[_position];
      if (_screenParam2 == (unsigned char)0x0E || _screenParam2 == (unsigned char)0x0F)
         {
         result = iFinish;
         // Valid Control code sequence
         switch (_screenParam1)
            {
            case (unsigned char)0x00 : _screenInterface->setCharSet(CS_NORMAL);
               break;
            case (unsigned char)0x02 : _screenInterface->setCharSet(CS_ALT1);
               break;
            case (unsigned char)0x04 : _screenInterface->setCharSet(CS_ALT2);
               break;
            default : result = iError;  //###Unsure of how to handle the error!
               break;
            }
         }
      else
         {
         result = iNone;
         }
      _screenParamLevel = PARAM1;
   break;
   default :
      _screenParamLevel = PARAM1; //###Unsure. Maybe remove to be able to show what
      // state the _screenParamLevel was in!
      result = iError;
   break;
   }
return (result);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::screenColours()
{
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::screenAttribs()
{
iResult result;

switch (_screenParamLevel)
   {
   case PARAM1 :
      _screenParam1 = _buffer[_position];
      _screenParamLevel = PARAM2;
      result = iContinue;
   break;
   case PARAM2 :
      _screenParam2 = _buffer[_position];
      _screenParamLevel = PARAM3;
      result = iContinue;
   break;
   case PARAM3 :
      _screenParam3 = _buffer[_position];
      if (_screenParam3 == (unsigned char)0x0E)
         {
         // Set the stored attribute and turn it on
         attrib_t attribs;
         attribs.Attrib1 = (attrib1_t)_screenParam1;
         attribs.Attrib2 = (attrib2_t)_screenParam2;
         _screenInterface->setStoredAttribs(attribs);
         _screenInterface->toggleAttribs(true);
         result = iFinish;
         }
      else if (_screenParam3 == (unsigned char)0x0F)
         {
         // If the last parameter is hex(0F) then select normal text attributes
         // This is a very sparsly documented behaviour of the attribute sequence.
         // Set the stored attributes to the normal intensity and toggle them on
         // Then set the stored attributes to the attributes set in this
         // sequence but don't turn them on
         attrib_t current_attribs;
         current_attribs.Attrib1 = ATTR_NORMAL1;
         current_attribs.Attrib2 = ATTR_NORMAL2;
         _screenInterface->setStoredAttribs(current_attribs);
         _screenInterface->toggleAttribs(true);
         // Set the stored attributes to the new attribute sequence
         attrib_t attribs;
         attribs.Attrib1 = (attrib1_t)_screenParam1;
         attribs.Attrib2 = (attrib2_t)_screenParam2;
         _screenInterface->setStoredAttribs(attribs);
         result = iFinish;
         }
      else
         {
         result = iNone;
         }
      _screenParamLevel = PARAM1;
   break;
   default :
      _screenParamLevel = PARAM1; //###Unsure. Maybe remove to be able to show what
      // state the _screenParamLevel was in!
      result = iError;
   break;
   }
return (result);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::screenCursorBlink()
{
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::screenScreenMode()
{
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::screenSelfId()
{
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::screenScreenWrite()
{
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::screenDrawBox()
{
iResult result;
static int x1=0, x2=0, y1=0, y2=0;
static bool gotbounds=false;

switch (_screenParamLevel)
   {
   case PARAM1 :
      // This should be the HEX(02) or HEX(0B)
      _boxControlFlag = false;
      gotbounds = false;
      _screenParam1 = _buffer[_position];
      if ((_screenParam1 == (unsigned char)0x02) || (_screenParam1 == (unsigned char)0x0B))
         {
         x1 = _screenInterface->getx();
         y1 = _screenInterface->gety();
         x2 = x1;
         y2 = y1;
         _screenInterface->beginDrawBox();
         _screenParamLevel = PARAM2;
         result = iContinue;
         }
      else
         {
         _screenParamLevel = PARAM1;
         result = iError;
         }
   break;
   case PARAM2 :
      _screenParam2 = _buffer[_position];
      switch (_screenParam2)
      {
      case (unsigned char)0x0B :
         _boxControlFlag = true;
         _screenInterface->drawTextLine(DIR_DOWN, (_screenParam1 == (unsigned char)0x0B));
         result = iContinue;
      break;
      case (unsigned char)0x0A :
         _screenInterface->moveCaret(DIR_DOWN);
         _screenInterface->drawTextLine(DIR_DOWN, (_screenParam1 == (unsigned char)0x0B));
         result = iContinue;
      break;
      case (unsigned char)0x0C :
         _screenInterface->moveCaret(DIR_UP);
         _screenInterface->drawTextLine(DIR_UP, (_screenParam1 == (unsigned char)0x0B));
         result = iContinue;
      break;
      case (unsigned char)0x08 :
         _screenInterface->moveCaret(DIR_LEFT);
         if (_boxControlFlag)
            {
            _screenInterface->drawTextLine(DIR_LEFT, (_screenParam1 == (unsigned char)0x0B));
            }
         result = iContinue;
      break;
      case (unsigned char)0x09 :
         _screenInterface->moveCaret(DIR_RIGHT);
         if (!_boxControlFlag)
            {
            _screenInterface->drawTextLine(DIR_RIGHT, (_screenParam1 == (unsigned char)0x0B));
            }
         result = iContinue;
      break;
      case (unsigned char)0x0F :
         _screenInterface->endDrawBox();
         _screenParamLevel = PARAM1;
         result = iFinish;
      break;
      default :
         _screenParamLevel = PARAM1;
         result = iError;
      break;
      }
      x2 = max(_screenInterface->getx(), x2);
      y2 = max(_screenInterface->gety(), y2);
   break;
   default :
      result = iError;
      _screenParamLevel = PARAM1;
   break;
   }
return (result);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::screenDrawGraphic()
{
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::screenInit()
{
iResult result;
switch (_screenParamLevel)
   {
   case PARAM1 :
      _screenParam1 = _buffer[_position];
      if (_screenParam1 == (unsigned char)0x0C)
         {
         _screenParamLevel = PARAM2;
         result = iContinue;
         }
      else
         {
         result = iNone;
         }
   break;
   case PARAM2 :
      _screenParam2 = _buffer[_position];
      if (_screenParam2 == (unsigned char)0x03)
         {
         _screenParamLevel = PARAM3;
         result = iContinue;
         }
      else
         {
         result = iNone;
         }
   break;
   case PARAM3 :
      if (_buffer[_position] == (unsigned char)0x0F)
         {
         _screenInterface->initScreen();
         _screenInterface->clearScreen();
         _screenParamLevel = PARAM1;
         result = iFinish;
         }
      else
         {
         result = iNone;
         }
   break;
   default :
      result = iError;
      _screenParamLevel = PARAM1;
   break;
   }
return (result);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::screenKeyboard()
{
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::screenFbEquiv()
{
}

//---------------------------------------------------------------------------
//
//    Level 2 Screen Functions
//
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::screenFtOther()
{
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::screenWtColour()
{
iResult result;

switch (_screenParamLevel)
   {
   case PARAM1 : //First parameter
      _screenParam1 = _buffer[_position];
      _screenParamLevel = PARAM2;
      result = iContinue;
   break;
   case PARAM2 : //second parameter
      _screenParam2 = _buffer[_position];
      _screenParamLevel = PARAM3;
      result = iContinue;
   break;
   case PARAM3 : //final parameter
      _screenParam3 = _buffer[_position];
      if ((_screenParam3 == (unsigned char)0x0E) || (_screenParam3 == (unsigned char)0x0F))
         {
         _screenInterface->setColour(CC_NORMAL_FORE,
            (colour_t)((_screenParam1 & (unsigned char)0xF0) >> 4));
         _screenInterface->setColour(CC_NORMAL_BACK,
            (colour_t)(_screenParam1 & (unsigned char)0x0F));
         _screenInterface->setColour(CC_UNDERLINE,
            (colour_t)((_screenParam2 & (unsigned char)0xF0) >> 4));
         _screenInterface->setColour(CC_REVERSE,
            (colour_t)(_screenParam2 & (unsigned char)0x0F));
         // See if we should turn off the standard attributes
         if (_screenParam3 == (unsigned char)0x0F)
            {
            _screenInterface->toggleAttribs(false);
            }
         result = iFinish;
         }
      else
         {
         result = iError;
         }
      _screenParamLevel = PARAM1;
   break;
   default :
      result = iError;
      _screenParamLevel = PARAM1;
   break;
   }
return (result);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::screenWtStatusWrite()
{
iResult result;

switch (_screenParamLevel)
   {
   case PARAM1 :
      if (_buffer[_position] == (unsigned char)0x0E)
         {
         _screenParamLevel = PARAM2;        // Can proceed to read parameters
         result = iContinue;
         }
      else
         {
         _screenParamLevel = PARAM1;
         result = iError;
         }
   break;
   case PARAM2 :  // Pick the attributes
      attrib_t attribs;
      switch (_buffer[_position])
         {
         case (unsigned char)0x80 : // Normal
            attribs.Attrib1 = ATTR_NORMAL1;
            attribs.Attrib2 = ATTR_NORMAL2;
         break;
         case (unsigned char)0x00 : // Bright
            attribs.Attrib1 = ATTR_BRIGHT;
            attribs.Attrib2 = ATTR_NORMAL2;
         break;
         case (unsigned char)0x82 : // Blinking
            attribs.Attrib1 = ATTR_BLINK;
            attribs.Attrib2 = ATTR_NORMAL2;
         break;
         case (unsigned char)0x84 : // Reverse
            attribs.Attrib1 = ATTR_NORMAL1;
            attribs.Attrib2 = ATTR_REVERSE;
         break;
         case (unsigned char)0x04 : // Reverse + Bright
            attribs.Attrib1 = ATTR_BRIGHT;
            attribs.Attrib2 = ATTR_REVERSE;
         break;
         case (unsigned char)0x86 : // Reverse + Blinking
            attribs.Attrib1 = ATTR_BLINK;
            attribs.Attrib2 = ATTR_REVERSE;
         break;
         case (unsigned char)0x02 : // Bright + Blinking
            attribs.Attrib1 = ATTR_BRIGHT_BLINK;
            attribs.Attrib2 = ATTR_NORMAL2;
         break;
         case (unsigned char)0x06 : // Bright + Blinking + Reverse
            attribs.Attrib1 = ATTR_BRIGHT_BLINK;
            attribs.Attrib2 = ATTR_REVERSE;
         break;
         default : // Invalid attribute code, default to Normal   //###Check this
            attribs.Attrib1 = ATTR_NORMAL1;
            attribs.Attrib2 = ATTR_NORMAL2;
         break;
         }
         _screenInterface->clearStatusLine();
         _screenInterface->setStatusLineAttribs(attribs);
         _screenParamLevel = PARAM3;
         result = iContinue;
   break;
   case PARAM3 :  // Keep outputting to the status line until a Carriage Return is found
      if ((_buffer[_position] == (unsigned char)0x0D) ||
         (_screenInterface->writeStatusLine(_buffer[_position], false)))
         {
         // Add the character to the text buffer
         // If the status line reaches the screen width, then stop
         // writing to the status line
         // No more text to display
         _screenInterface->finishStatusLine();
         _screenParamLevel = PARAM1;
         result = iFinish;
         }
      else
         {
         result = iContinue;
         }
   break;
   default :
      _screenParamLevel = PARAM1;
      result = iError;
   break;
   }
return (result);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::screenWtStatusClear()
{
if (_buffer[_position] == (unsigned char)0x0E)
   {
   //Restore the status line
   _screenInterface->clearStatusLine();
   return (iFinish);
   }
else
   {
   return (iError);
   }
}

//---------------------------------------------------------------------------
//
//    Level 2 Control Functions
//
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::controlDataByte()
{
_buffer[_position] = (unsigned char)0xFB;
return (iFinishContinue); //###Check
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::controlCompress()
{
iResult result;

switch (_controlParamLevel)
   {
   case PARAM1 : // Number of repeated characters
      _controlParam1 = _buffer[_position];
      _controlParamLevel = PARAM2;
      result = iContinue;
   break;
   case PARAM2 : // Character code to repeat
      int newSize, oldSize, repeatCnt;
      unsigned char *tempStr;

      // If we have a buffer like FB, 06, 41 then the resultant buffer should be
      // 41, 41, 41, 41, 41, 41
      _controlParam2 = _buffer[_position];
      oldSize = (_count - _position)-1;
      repeatCnt = int(_controlParam1);
      newSize = repeatCnt + oldSize;
      tempStr = new unsigned char[newSize];
      memset(tempStr, _controlParam2, max(0, repeatCnt));
      memcpy(&tempStr[repeatCnt], &_buffer[_position+1], max(0, oldSize));
      // Setup the new buffer properties
      _count = newSize;
      delete [] _buffer;
      _buffer = tempStr;
      _position = 0;
      _controlParamLevel = PARAM1;
      result = iFinishContinue; 
   break;
   }
return (result);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::controlRsdStart()
{
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::controlRsdAbort()
{
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::controlRsdGo()
{
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::controlRsdStop()
{
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::controlSelScreen()
{
// ### Flush the data to the printer or file
//_printerInterface->flush();
// Select the screen as the output device
_outDevice = DEV_SCREEN;

// Revert the FB sequence to last time the screen was selected
_controlFuncLevel1 = _scrControlFuncLevel1;
_controlFuncLevel2 = _scrControlFuncLevel2;
_controlParamLevel = _scrControlParamLevel;
_controlParam1 = _scrControlParam1;
_controlParam2 = _scrControlParam2;
_controlParam3 = _scrControlParam3;
_controlParam4 = _scrControlParam4;
// Clear the stored data
_scrControlFuncLevel1 = CONTROL1_CHECK;
_scrControlFuncLevel2 = CONTROL2_CHECK;
_scrControlParamLevel = PARAM1;
_scrControlParam1 = 0x00;
_scrControlParam2 = 0x00;
_scrControlParam3 = 0x00;
_scrControlParam4 = 0x00;

return (iContinue);

//_position--;
//return (iFinish); //###Modified for checking
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::controlSelPrinter()
{
// Select the printer as the output device
// ### Flush the data to the printer or file
//_printerInterface->flush();
_outDevice = DEV_PRINTER;

// Revert the FB sequence to last time the printer was selected
_controlFuncLevel1 = _prnControlFuncLevel1;
_controlFuncLevel2 = _prnControlFuncLevel2;
_controlParamLevel = _prnControlParamLevel;
_controlParam1 = _prnControlParam1;
_controlParam2 = _prnControlParam2;
_controlParam3 = _prnControlParam3;
_controlParam4 = _prnControlParam4;
// Clear the stored data
_prnControlFuncLevel1 = CONTROL1_CHECK;
_prnControlFuncLevel2 = CONTROL2_CHECK;
_prnControlParamLevel = PARAM1;
_prnControlParam1 = 0x00;
_prnControlParam2 = 0x00;
_prnControlParam3 = 0x00;
_prnControlParam4 = 0x00;

return (iContinue);

//_position--;
//return (iFinish); //###Modified for checking
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::controlResetTerm()
{
// Set the
//_printerInterface->flush();
_outDevice = DEV_SCREEN;
//_toggleOn = false; //##Check this
_screenParamLevel = PARAM1;
_controlParamLevel = PARAM1;
_printerParamLevel = PARAM1;
_screenFuncLevel1 = SCREEN1_CHECK;
_screenFuncLevel2 = SCREEN2_CHECK;
_screenFuncLevel3 = SCREEN3_CHECK;
_controlFuncLevel1 = CONTROL1_CHECK;
_controlFuncLevel2 = CONTROL2_CHECK;
_printerFuncLevel1 = PRINTER1_CHECK;
_printerFuncLevel2 = PRINTER2_CHECK;
_printerFuncLevel3 = PRINTER3_CHECK;
return (iFinish); //###Modified for checking
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::controlAuxParam()
{
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::controlCursorBlink2()
{
// Ignore at the moment, as we cannot blink or unblink the cursor
return (iFinish);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::controlAuxSwitch()
{
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::controlClearCrt()
{
return (iFinish);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::controlAuxFormat()
{
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::controlCursorSteady()
{
// Ignore at the moment, as we cannot blink or unblink the cursor
return (iFinish);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::controlCursorBlink3()
{
// Ignore at the moment, as we cannot blink or unblink the cursor
//_position--;
return (iFinish); //### Modified for checking
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::controlDelay()
{
// Extract the low order nibble
int tempInt = _buffer[_position] & 0x000F;
// If it's 1 to 9 then delay for 1/6 to 9/6 seconds
if ((tempInt > 0) && (tempInt < 10))
   {
   // (tempInt / 6)
   }
return (iFinish);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::controlNewFb()
{
if (_outDevice == DEV_SCREEN)
   {
   _scrControlFuncLevel1 = _controlFuncLevel1;
   _scrControlFuncLevel2 = _controlFuncLevel2;
   _scrControlParamLevel = _controlParamLevel;
   _scrControlParam1 = _controlParam1;
   _scrControlParam2 = _controlParam2;
   _scrControlParam3 = _controlParam3;
   _scrControlParam4 = _controlParam4;
   }
else if (_outDevice == DEV_PRINTER)
   {
   _prnControlFuncLevel1 = _controlFuncLevel1;
   _prnControlFuncLevel2 = _controlFuncLevel2;
   _prnControlParamLevel = _controlParamLevel;
   _prnControlParam1 = _controlParam1;
   _prnControlParam2 = _controlParam2;
   _prnControlParam3 = _controlParam3;
   _prnControlParam4 = _controlParam4;
   }
return (iFinish);
}
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// Level 1 Printer Functions
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::printerOutputChar()
{
_printerInterface->print(_buffer[_position]);
return (iFinish);
}

//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// Level 3 Printer Functions
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::printerWtCreateFile()
{
iResult result;

switch (_printerParamLevel)
   {
   case PARAM1 : // Start of the filename
      _fileInterface->close();
      if (_buffer[_position] != (unsigned char)0x0E)
         {
         _fileInterface->addToFileName(_buffer[_position]);
         _printerParamLevel = PARAM2;
         result = iContinue;
         }
      else
         {
         // No filename specified, so do not attempt to create the file!
         result = iFinish;
         }
   break;
   case PARAM2 : // The rest of the filename
      if (_buffer[_position] != (unsigned char)0x0E)
         {
         // character is part of the filename
         _fileInterface->addToFileName(_buffer[_position]);
         result = iContinue;
         }
      else
         {
         // Filename is finished and we can finish this routine
         if (_fileInterface->create() == '0')
            {
            // Failed to create the new file
            /// ### Possibly throw an exception or maybe the FileInterface
            // should display the error
            }
         _printerParamLevel = PARAM1;
         result = iFinish;
         }
   break;
   default :
      result = iError;
   break;
   }
return (result);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::printerWtTestFile()
{
iResult result;

switch (_printerParamLevel)
   {
   case PARAM1 : // Start of the filename
      _fileInterface->close();
      if (_buffer[_position] != (unsigned char)0x0E)
         {
         _fileInterface->addToFileName(_buffer[_position]);
         _printerParamLevel = PARAM2;
         result = iContinue;
         }
      else
         {
         // Check if the file exists (it should not as the filename is blank
         unsigned char exists = _fileInterface->exists();
         _commInterface->write(&exists, 1);
         result = iFinish;
         }
   break;
   case PARAM2 : // The rest of the filename
      if (_buffer[_position] != (unsigned char)0x0E)
         {
         // character is part of the filename
         _fileInterface->addToFileName(_buffer[_position]);
         result = iContinue;
         }
      else
         {
         // Filename is finished and we can finish this routine
         unsigned char exists = _fileInterface->exists();
         _commInterface->write(&exists, 1);
         _printerParamLevel = PARAM1;
         result = iFinish;
         }
   break;
   default :
      result = iError;
   break;
   }
return (result);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::printerWtReroute()
{
iResult result;

switch (_printerParamLevel)
   {
   case PARAM1 : // Start of the filename
      _printerInterface->setFileName("");
      if (_buffer[_position] != (unsigned char)0x0E)
         {
         _printerInterface->addToFileName(_buffer[_position]);
         _printerParamLevel = PARAM2;
         result = iContinue;
         }
      else
         {
         // No filename specified, so do not redirect printer output!
         result = iFinish;
         }
   break;
   case PARAM2 : // The rest of the filename
      if (_buffer[_position] != (unsigned char)0x0E)
         {
         // character is part of the filename
         _printerInterface->addToFileName(_buffer[_position]);
         result = iContinue;
         }
      else
         {
         // Filename is finished and we can finish this routine
         _printerInterface->switchToFile();
         _printerParamLevel = PARAM1;
         result = iFinish;
         }
   break;
   default :
      result = iError;
   break;
   }
return (result);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::printerWtRestore()
{
iResult result;

switch (_printerParamLevel)
   {
   case PARAM1 : // Waiting
      _printerParamLevel = PARAM2;
      result = iContinue;
   break;
   case PARAM2 : // This should finish the sequence
      if (_buffer[_position] == (unsigned char)0x0E)
         {
         _printerInterface->switchToPrinter();
         }
      _printerParamLevel = PARAM1;
      result = iFinish;
   break;
   default :

   break;
   }
return (result);
}

//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// LEVEL 1 HANDLING FUNCTIONS
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::handleControlCodes1()
{
//static control1_t _controlFuncLevel1 = CONTROL1_CHECK;
iResult result;

switch (_controlFuncLevel1)
   {
   case CONTROL1_CHECK :
      _controlFuncLevel1 = checkControlCode1(_buffer[_position]);
      //###Check the following if statement!
      if (_controlFuncLevel1 != CONTROL1_CHECK)
         {
         // Allowable code
         result = iContinue;
         }
      else
         {
         // Function code not matched
         result = iNone;
         }
   break;
   case FB :   // Handle the second level FB sequence
      result = handleControlCodes2();
      // if result == iNone, then the code can be checked by the next code
      // checking method
   break;
   default :   // Error!
      result = iError;
   break;
   }
// Ensure the state of the function is reset
if (result != iContinue)
   {
   _controlFuncLevel1 = CONTROL1_CHECK;
   }
return (result);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::handleScreenCodes1()
{
//static screen1_t _screenFuncLevel1 = SCREEN1_CHECK;
iResult result;

switch (_screenFuncLevel1)
   {
   case SCREEN1_CHECK :   // The current code in the buffer needs checking
      _screenFuncLevel1 = checkScreenCode1(_buffer[_position]);
      // Check for any codes to be immediately acted uppon
      switch (_screenFuncLevel1)
         {
         case NONE : // Null character
         case RESERVED1 : // Reserved
         case RESERVED2 : // Reserved
            // Do nothing
            result = iFinish;
         break;
         case HOME : // Move the cursor to the home position
            result = screenHome();
         break;
         case MULTI : // Start of a multibyte sequence
            result = iContinue;
         break;
         case CLEAR : // Clear the screen
            result = screenClear();
         break;
         case CURSOR_ON : // Turn the cursor on
            result = screenCursorOn();
         break;
         case CURSOR_OFF : // Turn the cursor off
            result = screenCursorOff();
         break;
         case ALARM : // Sound the beeper
            result = screenAlarm();
         break;
         case LEFT : // Move the text cursor to the left
            result = screenLeft();
         break;
         case RIGHT : // Move the text cursor to the right
            result = screenRight();
         break;
         case DOWN : // Move the cursor down
            result = screenDown();
         break;
         case UP : // Move the cursor up
            result = screenUp();
         break;
         case START_LINE : // Moves the cursor to the start of the current line
            result = screenStartLine();
         break;
         case ATTRIB_ON : // Turns the current text attribute on
            result = screenAttribOn();
         break;
         case ATTRIB_OFF : // Restores the current text attribute to Normal text
            result = screenAttribOff();
         break;
         case SCREEN1_OUTPUT : // Output the character
            result = screenOutputChar();
         break;
         default : // Error!
            result = iError;
         break;
         }
   break;
   case MULTI : // A multibyte sequence was found, now check which one it was
      result = handleScreenCodes2();
      // If a parameter was unknown, then set the state of the object up for
      // outputting the character next time
      if (result == iNone)
         {
         _position--;
         _screenFuncLevel1 = SCREEN1_OUTPUT;
         result = iContinue;
         }
      //else if (result != iContinue)
      //   {
         //propogate a fatal error back to the calling function??
      //   }
   break;
   case SCREEN1_OUTPUT :   //###May NOT need!?
      // If a parameter in the control sequence was not recognised, then
      // the result should propergate to this point and the unknown character
      // outputted to the screen. Thus, there should only be two places for
      // outputting a character to the screen.
      result = screenOutputChar();
   break;
   default : // Error
      result = iError;
   break;
   }
// Ensure the state of the function is reset
if (result != iContinue)
   {
   _screenFuncLevel1 = SCREEN1_CHECK;
   }
return (result);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::handlePrinterCodes1()
{
//static printer1_t _printerFuncLevel1 = PRINTER1_CHECK;
iResult result;

switch (_printerFuncLevel1)
   {
   case PRINTER1_CHECK :   // Need to check if the character is a function
      _printerFuncLevel1 = checkPrinterCode1(_buffer[_position]);
      if (_printerFuncLevel1 != PRINTER1_OUTPUT)
         {
         // Control code, so we can continue
         result = iContinue;
         }
      else
         {
         // Not a control code, so output the character
         result = printerOutputChar();
         }
   break;
   case CONTROL : // Control code that needs further checking
      result = handlePrinterCodes2();
      // If a parameter was unknown, then set the state of the object up for
      // outputting the character next time
      // ### Possibly directly output the character
      if (result == iNone)
         {
         _position--;
         _printerFuncLevel1 = PRINTER1_OUTPUT;
         result = iContinue;
         }
   break;
   case PRINTER1_OUTPUT : // Output the character
      result = printerOutputChar();
   break;
   default :
      result = iError; //### unsure!?
   break;
   }
// Ensure the state of the function is reset
if (result != iContinue)
   {
   _printerFuncLevel1 = PRINTER1_CHECK;
   }
return (result);
}

//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// LEVEL 2 HANDLING FUNCTIONS
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::handleControlCodes2()
{
iResult result;
control2_t tmpControlFuncLevel2;

//tmpControlFuncLevel2 = checkControlCode2(_buffer[_position]);
// Handle the case when we receive a new sequence of FB codes. That is,
// the first FB sequence is interrupted by a second sequence.
if (_buffer[_position] == NEW_FB)
   {
   controlNewFb();
   _controlFuncLevel2 = CONTROL2_CHECK;
   return (iContinue);
   }

switch (_controlFuncLevel2)
   {
   case CONTROL2_CHECK :
      _controlFuncLevel2 = checkControlCode2(_buffer[_position]);
      //tmpControlFuncLevel2 = checkControlCode2(_buffer[_position]);
      // Handle the case when we receive a new sequence of FB codes. That is,
      // the first FB sequence is interrupted by a second sequence.
      //if (tmpControlFuncLevel2 == NEW_FB)
      //   {
      //   controlNewFb();
      //   _controlFuncLevel2 = CONTROL2_CHECK;
      //   return (iContinue);
      //   }
      //else
      //   {
      //   _controlFuncLevel2 = tmpControlFuncLevel2;
      //   }
      // Check for any codes to be immediately acted uppon
      switch (_controlFuncLevel2)
         {
         case DATA_BYTE :  // WT/FT treat Hex(FB) as a data byte
            result = controlDataByte();
         break;
         case SEL_SCREEN : // WT/FT Select the screen for future data output
            result = controlSelScreen();
         break;
         case SEL_PRINTER :// WT/FT Select the printer for future data output
            result = controlSelPrinter();
         break;
         case RESET_TERM : // WT/FT Reset the terminal
            result = controlResetTerm();
         break;
         case CURSOR_BLINK2 : // WT?/FT Cursor blinking
            result = controlCursorBlink2();
         break;
         case CLEAR_CRT :  // FT Reset CRT ONLY, clear terminal data from input buffer
            result = controlClearCrt();
         break;
         case CURSOR_STEADY : // WT?/FT Cursor Steady
            result = controlCursorSteady();
         break;
         case CURSOR_BLINK3 : // WT?/FT Set the Cursor to blinking
            result = controlCursorBlink3();
         break;
         case DELAY :      // FT Delay any activity for the specified amount of time
            result = controlDelay();
         break;
         case CONTROL2_OUTPUT :
            // Above function should handle outputting the character
            result = iNone;
         break;
         default :
            // All other control functions need more processing
            result = iContinue;
         break;
         }
/*
      if (_controlFuncLevel2 != CONTROL2_OUTPUT)
         {
         // Allowable code
         result = iContinue;
         }
      else
         {
         // Unknown code. Above function should handle outputting the character
         result = iNone;
         } */
   break;
   // Other codes that require parameters
   case COMPRESS :   // WT/FT - Incoming data is compressed
      result = controlCompress();
   break;
   case RSD_START :  // FT start of Remote Screen Dump
      result = controlRsdStart();
   break;
   case RSD_ABORT :  // FT Abort the Remote Screen Dump
      result = controlRsdAbort();
   break;
   case RSD_GO :     // FT Remote screen Dump flow control - Go
      result = controlRsdGo();
   break;
   case RSD_STOP :   // FT Remote screen Dump flow control - Stop
      result = controlRsdStop();
   break;
   case AUX_PARAM :  // FT Setup the Auxilary port parameters
      result = controlAuxParam();
   break;
   case AUX_SWITCH : // FT Enable/Disable Auxilary port
      result = controlAuxSwitch();
   break;
   case AUX_FORMAT : // FT Set data transfer format for Auxilary ports
      result = controlAuxFormat();
   break;
   default :
      result = iNone;
   break;
   }
// Ensure the state of the function is reset
if (result != iContinue)
   {
   _controlFuncLevel2 = CONTROL2_CHECK;
   }
return (result);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::handleScreenCodes2()
{
//static screen2_t _screenFuncLevel2 = SCREEN2_CHECK;
iResult result;

switch (_screenFuncLevel2)
   {
   case SCREEN2_CHECK : // Find out what function is being used
      _screenFuncLevel2 = checkScreenCode2(_buffer[_position]);
      if (_screenFuncLevel2 != SCREEN2_OUTPUT)
         {
         // Allowable code
         result = iContinue;
         }
      else
         {
         // Unknown code. Calling function should handle outputting the character
         result = iNone;
         }
   break;
   case ZERO :       // WT/FT 3rd level function
      result = screenZero();
   break;
   case OPEN_WIN :   // FT Open window function
      result = screenOpenWin();
   break;
   case CHAR_SET :   // WT/FT Character set selection
      result = screenCharSet();
   break;
   case COLOURS :    // FT Set active colours
      result = screenColours();
   break;
   case ATTRIBS :    // WT/FT Current character attributes
      result = screenAttribs();
   break;
   case CURSOR_BLINK :  // FT Cursor blinking
      result = screenCursorBlink();
   break;
   case SCREEN_MODE :   // FT Set screen mode (columns and lines)
      result = screenScreenMode();
   break;
   case SELF_ID :       // FT Self ID request
      result = screenSelfId();
   break;
   case SCREEN_WRITE :  // FT Entire Screen Read/Write
      result = screenScreenWrite();
   break;
   case DRAW_BOX :      // WT/FT Box drawing function
      result = screenDrawBox();
   break;
   case DRAW_GRAPHIC :  // FT Graphics drawing function
      result = screenDrawGraphic();
   break;
   case INIT :          // WT/FT Screen Initialization
      result = screenInit();
   break;
   case KEYBOARD :      // FT Keyboard selection
      result = screenKeyboard();
   break;
   case FB_EQUIV :      // FT Hex(FB) equivalent sequence
      result = screenFbEquiv();
   break;
   default :            // Default to re-checking a character in Level1
      result = iNone;
   break;
   }
// Ensure the state of the function is reset
if (result != iContinue)
   {
   _screenFuncLevel2 = SCREEN2_CHECK;
   }
return (result);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::handlePrinterCodes2()
{
//static printer2_t _printerFuncLevel2 = PRINTER2_CHECK;
iResult result;

switch (_printerFuncLevel2)
   {
   case PRINTER2_CHECK :
      _printerFuncLevel2 = checkPrinterCode2(_buffer[_position]);
      if (_printerFuncLevel2 != PRINTER2_OUTPUT)
         {
         // Allowable code
         result = iContinue;
         }
      else
         {
         // Unknown code. Calling function should handle outputting the character
         result = iNone;
         }
   break;
   case PRINTER_ZERO :
      result = handlePrinterCodes3();
   break;
   default :
      result = iError;  // ###Unsure!
   break;
   }
// Ensure the state of the function is reset
if (result != iContinue)
   {
   _printerFuncLevel2 = PRINTER2_CHECK;
   }
return (result);
}

//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// LEVEL 3 HANDLING FUNCTIONS
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
iResult WtInterpreter::handlePrinterCodes3()
{
//static printer3_t _printerFuncLevel3 = PRINTER3_CHECK;
iResult result;

switch (_printerFuncLevel3)
   {
   case PRINTER3_CHECK :
      _printerFuncLevel3 = checkPrinterCode3(_buffer[_position]);
      if (_printerFuncLevel3 != PRINTER3_OUTPUT)
         {
         // Allowable code
         result = iContinue;
         }
      else
         {
         // Unknown code. Calling function should handle outputting the character
         result = iNone;
         }
   break;
   case CREATE_FILE :
      result = printerWtCreateFile();
   break;
   case TEST_FILE :
      result = printerWtTestFile();
   break;
   case REROUTE :
      result = printerWtReroute();
   break;
   case RESTORE :
      result = printerWtRestore();
   break;
   default :
      result = iError;
   break;
   }
// Ensure the state of the function is reset
if (result != iContinue)
   {
   _printerFuncLevel3 = PRINTER3_CHECK;
   }
return (result);
}

//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// LEVEL 1 CHECKING FUNCTIONS
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
control1_t WtInterpreter::checkControlCode1(unsigned char code)
{
switch (code)
   {
   case FB :
      return (FB);
   default :
      return (CONTROL1_CHECK);
   }
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
screen1_t WtInterpreter::checkScreenCode1(unsigned char code)
{
switch (code)
   {
   case NONE : // Null character
   case RESERVED1 : // Reserved
   case RESERVED2 : // Reserved
   case HOME : // Move the cursor to the home position
   case MULTI : // Start of a multibyte sequence
   case CLEAR : // Clear the screen
   case CURSOR_ON : // Turn the cursor on
   case CURSOR_OFF : // Turn the cursor off
   case ALARM : // Sound the beeper beep
   case LEFT : // Move the text cursor to the left
   case RIGHT : // Move the text cursor to the right
   case DOWN : // Move the cursor down
   case UP : // Move the cursor up
   case START_LINE : // Moves the cursor to the start of the current line
   case ATTRIB_ON : // Turns the current text attribute on
   case ATTRIB_OFF : // Restores the current text attribute to Normal text
      return ((screen1_t)code);
   default : // No control character found
      return (SCREEN1_OUTPUT);
   }
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
printer1_t WtInterpreter::checkPrinterCode1(unsigned char code)
{
if (code == CONTROL)
   return (CONTROL);
else
   return (PRINTER1_OUTPUT);
}

//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// LEVEL 2 CHECKING FUNCTIONS
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
control2_t WtInterpreter::checkControlCode2(unsigned char code)
{
if (code > (unsigned char)0x03 && code <= (unsigned char)0x5F)
   {
   _position--; // since the first parameter is a character within this range
      // allow the decompression routine access to it next time it loops through
   return (COMPRESS);
   }               
if ((code >= (unsigned char)0xC1) && (code <= (unsigned char)0xC9))
   {
   return (DELAY);
   }
switch (code)
   {
   case DATA_BYTE :
   case RSD_START :
   case RSD_ABORT :
   case RSD_GO :
   case RSD_STOP :
   case SEL_SCREEN :
   case SEL_PRINTER :
   case RESET_TERM :
   case AUX_PARAM :
   case CURSOR_BLINK2 :
   case AUX_SWITCH :
   case CLEAR_CRT :
   case AUX_FORMAT :
   case CURSOR_STEADY :
   case CURSOR_BLINK3 :
   case NEW_FB :
      return ((control2_t)code);
   default :
      return (CONTROL2_OUTPUT);
   }
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
screen2_t WtInterpreter::checkScreenCode2(unsigned char code)
{
switch (code)
   {
   case ZERO :       // WT/FT 3rd level function
   case OPEN_WIN :   // FT Open window function
   case CHAR_SET :   // WT/FT Character set selection
   case COLOURS :    // FT Set active colours
   case ATTRIBS :    // WT/FT Current character attributes
   case CURSOR_BLINK :  // FT Cursor blinking
   case SCREEN_MODE :   // FT Set screen mode (columns and lines)
   case SELF_ID :       // FT Self ID request
   case SCREEN_WRITE :  // FT Entire Screen Read/Write
   case DRAW_BOX :      // WT/FT Box drawing function
   case DRAW_GRAPHIC :  // FT Graphics drawing function
   case INIT :          // WT/FT Screen Initialization
   case KEYBOARD :      // FT Keyboard selection
   case FB_EQUIV :      // FT Hex(FB) equivalent sequence
      return ((screen2_t)code);
   default :            // Default to re-checking a character in Level1
      return (SCREEN2_OUTPUT);
   }
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
printer2_t WtInterpreter::checkPrinterCode2(unsigned char code)
{
if (code == PRINTER_ZERO)
   return (PRINTER_ZERO);
else
   return (PRINTER2_OUTPUT);
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
printer3_t WtInterpreter::checkPrinterCode3(unsigned char code)
{
switch (code)
   {
   case CREATE_FILE :  // Create a file on the terminal
   case TEST_FILE :    // Test for the existence of a file
   case REROUTE :      // Reroute the printer output to a file
   case RESTORE :      // Restore the printing output device to the previous state
      return ((printer3_t)code);
   default :
      return (PRINTER3_OUTPUT);
   }
}

//---------------------------------------------------------------------------
// Method: WtInterpreter()
// Desc.: Default Constructor
// Return: None
// Notes:
//---------------------------------------------------------------------------
WtInterpreter::WtInterpreter() : Interpreter()
{
_canInterpret = false;
_toggleOn = false;
_boxControlFlag = false;
_screenInterface = 0;
_fileInterface = 0;
_printerInterface = 0;
_commInterface = 0;
_position = 0;
_buffer = 0;
// Initialize the parameter levels
_screenParamLevel = PARAM1;
_controlParamLevel = PARAM1;
_printerParamLevel = PARAM1;
// Initialize the function levels
_screenFuncLevel3 = SCREEN3_CHECK;
_screenFuncLevel1 = SCREEN1_CHECK;
_screenFuncLevel2 = SCREEN2_CHECK;
_controlFuncLevel1 = CONTROL1_CHECK;
_controlFuncLevel2 = CONTROL2_CHECK;

//
_scrControlFuncLevel1 = CONTROL1_CHECK;
_scrControlFuncLevel2 = CONTROL2_CHECK;
_prnControlFuncLevel1 = CONTROL1_CHECK;
_prnControlFuncLevel2 = CONTROL2_CHECK;
_scrControlParamLevel = PARAM1;
_prnControlParamLevel = PARAM1;

_printerFuncLevel1 = PRINTER1_CHECK;
_printerFuncLevel2 = PRINTER2_CHECK;
_printerFuncLevel3 = PRINTER3_CHECK;
_outDevice = DEV_SCREEN;
}

//---------------------------------------------------------------------------
// Method: interpret(buffer, count)
// Desc.: Reads through the buffer
// Return: true on success and false if a fatal error occured
// Notes: 'count' MUST be within the buffer's capacity
//---------------------------------------------------------------------------
bool WtInterpreter::interpret(const unsigned char *buffer, int count)
{
iResult controlResult;

if (_canInterpret)
   {
   // Setup the internal buffer
   _count = count;
   _buffer = new unsigned char[_count];
   memcpy(_buffer, buffer, _count);
   // Loop through each character in the buffer and interpret it
   for (_position=0; _position < _count; _position++)
      {
      // Check for the Flow of Control codes first
      // If no Flow of Control codes are found, then continue processing
      // the current code.
      // -Need to capture errors in this section. Any returned iError values
      // should be written to a log file with ten bytes of the buffer on either
      // side of the current position written to the file + the state of the
      // system.
      controlResult = handleControlCodes1();
      if (controlResult == iNone || controlResult == iFinishContinue) // || controlResult == iFinish
         {
         if (_outDevice == DEV_PRINTER)
            {
            // If the current output device is the printer, then check for printer
            // control codes
            handlePrinterCodes1();
            }
         else  // add-in check for other devices in future versions
            {
            // If the current output device is the screen, then check for screen
            // control codes
            handleScreenCodes1();
            }
         }
      }
   // Destroy the internal buffer
   delete [] _buffer;
   return (true);
   }
else
   {
   // Not all of the modules have been linked yet!
   return (false);
   }
}

//---------------------------------------------------------------------------
// Method: setFlowOn()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void WtInterpreter::setFlowOn()
{
Interpreter::setFlowOn();
_commInterface->write(FLOW_ON_STR, strlen(FLOW_ON_STR));
}

//---------------------------------------------------------------------------
// Method: setFlowOff()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------
void WtInterpreter::setFlowOff()
{
Interpreter::setFlowOff();
_commInterface->write(FLOW_OFF_STR, strlen(FLOW_OFF_STR));
}

//---------------------------------------------------------------------------
// Method: setFileInterface(fileInterface)
// Desc.: Links the internal file interface to an already created
//    FileInterface class
// Return:
// Notes:
//---------------------------------------------------------------------------
void WtInterpreter::setFileInterface(FileInterface *fileInterface)
{
if (fileInterface != 0)
   {
   _fileInterface = fileInterface;
   checkStatus(); // Check to see if the Interpreter can now interpret a
                  // buffer
   }
}

//---------------------------------------------------------------------------
// Method: setScreenInterface(screenInterface)
// Desc.: Links the internal screen interface to an already created
//    ScreenInterface Object
// Return:
// Notes:
//---------------------------------------------------------------------------
void WtInterpreter::setScreenInterface(ScreenInterface *screenInterface)
{
if (screenInterface != 0)
   {
   _screenInterface = screenInterface;
   checkStatus(); // Check to see if the Interpreter can now interpret a
                  // buffer
   }
}

//---------------------------------------------------------------------------
// Method: setCommInterface(commInterface)
// Desc.: Links the internal comm interface to an already created
//    CommInterface Object
// Return:
// Notes:
//---------------------------------------------------------------------------
void WtInterpreter::setCommInterface(CommInterface *commInterface)
{
if (commInterface != 0)
   {
   _commInterface = commInterface;
   checkStatus(); // Check to see if the Interpreter can now interpret a
                  // buffer
   }
}

//---------------------------------------------------------------------------
// Method: setPrinterInterface(printerInterface)
// Desc.: Links the internal printer interface to an already created
//    PrinterInterface Object
// Return:
// Notes:
//---------------------------------------------------------------------------
void WtInterpreter::setPrinterInterface(PrinterInterface *printerInterface)
{
if (printerInterface != 0)
   {
   _printerInterface = printerInterface;
   checkStatus(); // Check to see if the Interpreter can now interpret a
                  // buffer
   }
}

//---------------------------------------------------------------------------
// Method: ()
// Desc.:
// Return:
// Notes:
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// END OF FILE
//---------------------------------------------------------------------------


